import { AfterViewInit, ChangeDetectorRef, Component, OnInit, ViewChild } from '@angular/core';
import { BreakpointObserver } from '@angular/cdk/layout';
import { MatSidenav } from '@angular/material/sidenav';
import { delay, filter } from 'rxjs/operators';
import { NavigationEnd, Router } from '@angular/router';
import { UntilDestroy, untilDestroyed } from '@ngneat/until-destroy';
import { CommonService } from 'src/Services/common.service';
import { AuthService } from 'src/app/auth/auth.service';

@UntilDestroy()
@Component({
  selector: 'app-responsive-sidenav-admin',
  templateUrl: './responsive-sidenav-admin.component.html',
  styleUrls: ['./responsive-sidenav-admin.component.scss'],
})
export class ResponsiveSidenavAdminComponent implements AfterViewInit, OnInit {
  @ViewChild(MatSidenav)
  sidenav!: MatSidenav;

  menuOpen: boolean = false;
  userLogin: any;
  buttonState: boolean = false;
  logoState: boolean = true;

  // activeLink: string = '/';
  activeLinks: string[] = [];

  constructor(
    private observer: BreakpointObserver,
    private router: Router,
    private cdr: ChangeDetectorRef,private commonService: CommonService,
    private authService: AuthService,
  ) {
    // Listen to route changes to update the active link
    this.router.events.subscribe(event => {
      if (event instanceof NavigationEnd) {
        this.updateActiveLinks();
      }
    });
    this.commonService.popCloseEvent.subscribe((value) => {
      this.userLogin = value?.firstName + value?.lastName;
      this.buttonState = true;
      this.logoState = false;
    });
  }

  ngOnInit() {
    this.checkLoginStatus();
  }

  ngAfterViewInit() {
    this.observer
      .observe(['(max-width: 800px)'])
      .pipe(untilDestroyed(this))
      .subscribe((res) => {
        //@ts-ignore
        if (res.matches) {
          this.sidenav.mode = 'over';
          this.sidenav.close();
        } else {
          this.sidenav.mode = 'side';
          this.sidenav.open();
        }
        this.cdr.detectChanges(); // Manually trigger change detection
      });

    this.router.events
      .pipe(
        untilDestroyed(this),
        filter((e) => e instanceof NavigationEnd)
      )
      .subscribe(() => {
        if (this.sidenav.mode === 'over') {
          this.sidenav.close();
        }
      });
  }

  setActiveLinks(links: string[]): void {
    this.activeLinks = links;
  }

  isActive(link: string): boolean {
    return this.activeLinks.includes(link);
  }

  updateActiveLinks(): void {
    const currentUrl = this.router.url;

    if (currentUrl === '/administrator-list' || currentUrl === '/administrator-onboarding' || currentUrl === '/administrator-details') {
      this.activeLinks = ['/administrator-list', '/administrator-onboarding', '/administrator-details'];
    } else if (currentUrl === '/member-list' || currentUrl === '/member-onboarding' || currentUrl === '/member-details') {
      this.activeLinks = ['/member-list', '/member-onboarding', '/member-details'];
    } else if (currentUrl === '/trader-list' || currentUrl === '/trading-tokens' || currentUrl === '/trader-details'|| currentUrl === '/trading-account-details') {
      this.activeLinks = ['/trader-list', '/trading-tokens', '/trader-details', '/trading-account-details'];
    } else if (currentUrl === '/trusted-source-list' || currentUrl === '/trusted-source-onboarding' || currentUrl === '/trusted-source-details') {
      this.activeLinks = ['/trusted-source-list', '/trusted-source-onboarding', '/trusted-source-details'];
    } else if (currentUrl === '/ticket') {
      this.activeLinks = ['/ticket'];
    } else {
      this.activeLinks = [currentUrl];
    }
  }

  checkLoginStatus() {
    const token = this.authService.getToken();
    if (token) {
      this.buttonState = true;
      this.logoState = false;
    } else {
      this.buttonState = false;
      this.logoState = true;
    }
  }

  redirectToLogin() {
    this.router.navigate(['/login']); // Redirect to login page
  }

  SignOut() {
    this.authService.clearToken();
    this.logoState = true;
    this.buttonState = false;
    this.router.navigate(['/']); // Redirect to home page or login page
  }

  navigateTo(page: string) {
    if (this.buttonState) {
      this.router.navigate([page]);
    } else {
      this.redirectToLogin();
    }
  }
}