import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { Router } from '@angular/router';
import { title } from 'process';

@Component({
  selector: 'app-asset-publishment',
  templateUrl: './asset-publishment.component.html',
  styleUrls: ['./asset-publishment.component.scss'],
})
export class AssetPublishmentComponent implements OnInit, OnDestroy {
  useMockup: boolean = false;
  
  private accounts: any = null;
  private defaultDuration: number = 3000;
  protected txHash: any;

  constructor(
    private snackBar: MatSnackBar,
    private metamaskService: MetamaskChainService,
    private provenanceApiService: ProvenanceApiService,
    private clipboardManager: Clipboard,
    private router: Router
  ) {
    this.connectToMetamask();
  }

  @ViewChild('assetForm') assetForm!: NgForm;
  asset = {
    title: '',
    dcterms_title: '',
    dcterms_description_short: '',
    dcterms_type: '',
    dcat_endpointRLS: '',
    dcat_landingPage: '',
    dcat_keyword: '',
    dcterms_conformsTo: '',
    dcterms_description_long: '',
    fame_logo: '',
  };
  types: string[] = [
    'dataset',
    'model',
    'document',
    'service',
    'software',
    'image',
    'audio',
    'video',
    'other',
  ];
  isMetamaskConnected: boolean = false;
  isSubmitted: any = false;
  assetId: any = '';
  title: any;
  copyToClipboard: any;
  subscribeReq: any;
  subscribePt: any;

  ngOnInit() {
    localStorage.removeItem('offering');
  }

  ngOnDestroy(): void {
    if (this.subscribePt) this.subscribePt.unsubscribe();
    if (this.subscribeReq) this.subscribeReq.unsubscribe();
  }

  onDiscard() {
    this.assetForm.resetForm();
    this.router.navigate(['/']);
  }

  onSubmit() {
    if (this.assetForm.valid) {
      if (!this.asset.title) this.asset.title = crypto.randomUUID(); //TODO Title
      console.log('Form Submitted', this.asset);
      this.asset.dcterms_title = this.asset.title;
      this.title = this.asset.title;
      // Implement navigation logic
      this.subscribePt = this.provenanceApiService
        .submitAsset(this.asset)
        .subscribe((data: any) => {
          if (data) {
            console.log('Asset Given', data);
            // Implement navigation logic
            this.assetId = data.aid;
            this.metamaskService
              .postAsset(data, this.accounts[0])
              .then((transaction: any) => {
                if (transaction) {
                  console.log('Asset Created', transaction);
                  this.txHash = transaction.transactionHash;
                  this.snackBar.open(
                    `Transaction sent with ID: ${this.txHash}`,
                    'Info',
                    {
                      duration: this.defaultDuration,
                    }
                  );
                  localStorage.setItem('asset', JSON.stringify(this.asset));
                  this.assetForm.resetForm();
                  this.isSubmitted = true;
                  //TODO Implement navigation logic
                }
              })
              .catch((error: any) => {
                this.snackBar.open(JSON.stringify(error), 'Error', {
                  duration: this.defaultDuration,
                });
                this.isSubmitted = false;
              });
          }
        });
    } else {
      this.snackBar.open('Please fill all the required fields', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  public async connectToMetamask() {
    // @ts-ignore
    if (typeof window.ethereum === 'undefined' || !window.ethereum) {
      this.snackBar.open(
        'MetaMask plugin is not installed!',
        'Please install MetaMask',
        {
          duration: this.defaultDuration,
        }
      );
      this.isMetamaskConnected = false;
    } else {
      await this.getMetamaskAccounts();
      if (this.accounts == null) return;
      this.isMetamaskConnected = true;
      this.snackBar.open('MetaMask is connected!', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private async getMetamaskAccounts() {
    try {
      // @ts-ignore
      const accounts = await window.ethereum.request({
        method: 'eth_requestAccounts',
      });
      console.log('Accounts: ' + accounts);
      this.accounts = accounts;
    } catch (err) {
      // @ts-ignore
      if (err.code === 4001) {
        // EIP-1193 userRejectedRequest error
        // If this happens, the user rejected the connection request.
        // this.showToast(this.toastService, 'Please connect to MetaMask.', 'warning');
        this.snackBar.open('Please connect to MetaMask.', 'Warn', {
          duration: this.defaultDuration,
        });
      } else {
        this.snackBar.open(err + '', 'Error', {
          duration: this.defaultDuration,
        });
      }
    }
  }

  copyAssetUri(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.assetId);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    this.subscribeReq = this.provenanceApiService.checkRequestStatus(this.assetId).subscribe(
      (response: any) => {
        if (!response) return;
        console.log('Response', response);

        const message =
          response.message +
          ' at ' +
          response.finalized +
          ' with status ' +
          response.status;
        this.snackBar.open(message, 'Request status', {
          duration: 6000,
        });
      },
      (error) => {
        console.log('Error', error);
        this.snackBar.open(JSON.stringify(error), 'Error', {
          duration: this.defaultDuration,
        });
      }
    );
  }

  goToFdac($event: any) {
    $event.preventDefault();
    this.snackBar.open('Redirecting to FDAC...', 'Info', {
      duration: this.defaultDuration,
    });
    // Implement navigation logic
    // this.router.navigate(['/asset/' + this.assetId]);
    /*  window.open(
      'https://fame-fdac.iot-catalogue.com/components/' + this.assetId,
      '_blank'
    ); */
    window.open('http://localhost:4200/asset/' + this.assetId, '_blank');
  }

  goToOffering($event: any) {
    $event.preventDefault();
    this.router.navigate(['/offering-definition'], {
      queryParams: { aid: this.assetId, title: encodeURI(this.title) },
    });
  }
}
