import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { privateDecrypt } from 'crypto';
import { skipLast, subscribeOn } from 'rxjs';
import { CdkFixedSizeVirtualScroll } from '@angular/cdk/scrolling';
import { off } from 'process';
import { ActivatedRoute, Router } from '@angular/router';

@Component({
  selector: 'app-offering-definition',
  templateUrl: './offering-definition.component.html',
  styleUrls: ['./offering-definition.component.scss'],
})
export class OfferingDefinitionComponent implements OnInit, OnDestroy {
  private defaultDuration: number = 3000;
  protected txHash: any;
  offeringId: any = '';
  subscribePt: any;
  subscribeReq: any;

  constructor(
    private snackBar: MatSnackBar,
    private provenanceApiService: ProvenanceApiService,
    private clipboardManager: Clipboard,
    private route: ActivatedRoute,
    private router: Router
  ) {}

  @ViewChild('offeringForm') offeringForm!: NgForm;
  offering = {
    assetId: '',
    assetName: '',
    title: '',
    price: 0,
    summary: '',
    scope: '',
    cap: {
      subscription: 0,
      subscriptionType: '',
      downloads: 0,
      volume: 0,
      volumeQty: '',
    },
    license: '',
    sla: '',
    beneficiary: '',
    cdsTarget: '',
    cdsSl: '',
  };
  capTypes: string[] = ['subscription', 'downloads', 'volume'];
  subscriptionTypes: string[] = ['minute', 'hour', 'week', 'month', 'year'];
  volumeTypes: string[] = ['B', 'KB', 'MB', 'GB', 'TB'];
  isSubmitted: boolean = false;
  isProcessed: boolean = false;
  asset: any;
  copyToClipboard: any;
  selectedCap: any;

  ngOnInit() {
    this.isProcessed = false;
    this.isSubmitted = false;
    const offeringStored = localStorage.getItem('offering') || '{}';
    if (offeringStored !== '{}') this.offering = JSON.parse(offeringStored);
    this.route.queryParams.subscribe((params) => {
      console.log('Params', params);
      if (!params['aid'] && !params['price']) return;
      this.offering.assetId = params['aid'];
      if (this.offering.assetId)
        this.offering.assetId = decodeURI(this.offering.assetId);
      if (params['title']) this.offering.assetName = decodeURI(params['title']);
      if (params['price']) this.offering.price = params['price'];
    });
  }

  ngOnDestroy(): void {
    if (this.subscribePt) this.subscribePt.unsubscribe();
    if (this.subscribeReq) this.subscribeReq.unsubscribe();
  }

  onDiscard() {
    this.offeringForm.resetForm();
    localStorage.removeItem('offering');
    this.router.navigate(['/']);
  }

  onSubmit() {
    if (this.offeringForm.valid) {
      const offering = this.createOfferingSubmit();
      this.isSubmitted = true;
      this.subscribePt= this.provenanceApiService.submitOffering(offering).subscribe(
        (response) => {
          if (!response) return;
          // @ts-ignore
          // this.showToast(this.toastService, 'Offering confirmed with ID ' + JSON.stringify(response), 'info');
          this.snackBar.open(
            'Offering confirmed with ID ' + JSON.stringify(response),
            'Info',
            {
              duration: this.defaultDuration,
            }
          );
          this.isProcessed = true;
          this.offeringId = response;
          this.provenanceApiService.confirmOffering(this.offeringId).subscribe(
            (response) => {
              console.log('Offering confirmed');
            },
            (error) => {
              console.log('Error Offering confirmed', error);
              this.snackBar.open(JSON.stringify(error), 'Error', {
                duration: this.defaultDuration,
              });
            }
          );
          localStorage.removeItem('offering');
        },
        (error) => {
          //this.showToast(this.toastService, DashboardUtils.formatErrorMessage(error), 'danger');
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        }
      );
    } else {
      this.snackBar.open('Please fill all the required fields', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private createOfferingSubmit() {
    const offeringToSubmit = JSON.parse(JSON.stringify(this.offering));
    if (this.selectedCap === 'subscription') {
      offeringToSubmit.cap.subscription =
        offeringToSubmit.cap.subscription +
        offeringToSubmit.cap.subscriptionType;
    } else if (this.selectedCap === 'downloads') {
      offeringToSubmit.cap.downloads = offeringToSubmit.cap.downloads;
    } else if (this.selectedCap === 'volume') {
      offeringToSubmit.cap.volume =
        offeringToSubmit.cap.volumeQty + offeringToSubmit.cap.volume;
    }
    const offeringSubmit = {
      offering: offeringToSubmit,
      tid: offeringToSubmit.beneficiary,
      target: offeringToSubmit.cdsTarget,
      sl: offeringToSubmit.cdsSl,
    };
    return offeringSubmit;
  }

  copyUri(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.offeringId);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  goToFdac($event: any) {
    $event.preventDefault();
    this.snackBar.open('Redirecting to Details...', 'Info', {
      duration: this.defaultDuration,
    });
    // Implement navigation logic
    //this.router.navigate(['/asset/' + this.offeringId]);
    /*  window.open(
      'https://fame-fdac.iot-catalogue.com/components/' + this.offeringId,
      '_blank'
    ); */
    this.router.navigate(['/offering-details/' + this.offeringId]);
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    this.subscribeReq = this.provenanceApiService.checkRequestStatus(this.offeringId).subscribe(
      (response: any) => {
        if (!response) return;
        console.log('Response', response);
        const message =
        response.message +
        ' at ' +
        response.finalized +
        ' with status ' +
        response.status;
        this.snackBar.open(message, 'Request status', {
          duration: 6000,
        });
      },
      (error) => {
        console.log('Error', error);
        this.snackBar.open(JSON.stringify(error), 'Error', {
          duration: this.defaultDuration,
        });
      }
    );
  }

  openPat(event: any) {
    event.preventDefault();
    localStorage.setItem('offering', JSON.stringify(this.offering));
    this.router.navigate(['/pat'], {
      queryParams: { oid: this.offeringId, aid: this.offering.assetId },
    });
  }
}
