import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { NgForm } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { ActivatedRoute, Router } from '@angular/router';
import { title } from 'process';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { throwToolbarMixedModesError } from '@angular/material/toolbar';

@Component({
  selector: 'app-offering-details',
  templateUrl: './offering-details.component.html',
  styleUrls: ['./offering-details.component.scss'],
})
export class OfferingDetailstComponent implements OnInit, OnDestroy {
  useMockup: boolean = true;

  private accounts: any = null;
  private defaultDuration: number = 3000;
  protected txHash: any;
  access: any;
  duration: any;
  subscription: any;
  isSubmitted: any;
  offeringDetailsId: any;
  subscribeTm: any;

  constructor(
    private snackBar: MatSnackBar,
    private metamaskService: MetamaskChainService,
    private provenanceApiService: ProvenanceApiService,
    private tmApiService: TmApiService,
    private router: Router,
    private route: ActivatedRoute,
    private clipboardManager: Clipboard
  ) {
    this.connectToMetamask();
  }

  isMetamaskConnected: boolean = false;
  offeringId: any = '';
  assetName: any = '';
  offering: any;
  cap: string = '';
  capAdditional: string = '';
  capType: string = '';

  subscriptionTypes: string[] = ['m', 'H', 'W', 'M', 'V'];
  volumeTypes: string[] = ['B', 'KB', 'MB', 'GB', 'TB'];

  overviewItems: any = [
    /*  {
      title: 'Total Active Accounts',
      value: 10,
      description: '2 this month',
      isPositive: true,
      icon: '../../../assets/icons/ta_accounts.png'
    }, */
  ];

  ngOnInit() {
    //this.route.queryParams.subscribe((params) => {
    /*  const oid = this.route.snapshot.paramMap.get('id');
    if (!oid) return;
    this.offeringId = oid; */
    this.route.queryParams.subscribe((params) => {
      console.log('Params', params);
      if (!params['oid'] && !params['title']) {
        alert('Error: No offering id and asset title in URL');
        this.router.navigate(['/']);
        return;
      }
      this.offeringId = params['oid'];
      if (params['title']) this.assetName = decodeURI(params['title']);
    });
    this.provenanceApiService.getOfferingDetails(this.offeringId).subscribe(
      (data: any) => {
        //console.log('Data', data);
        if (data) {
          this.offering = data?.offering;
          this.decodeCap(this.offering);
          this.overviewItems.push({
            title: this.offering.title,
            value: this.offering.price,
            description:
              this.cap + ' (' + this.capAdditional + ' ' + this.capType + ')',
            isPositive: true,
            icon: '../../../assets/icons/ta_accounts.png',
          });
        }
      },
      (error) => {
        console.log('Error', error);
        this.snackBar.open('Error fetching offering details', 'Error', {
          duration: this.defaultDuration,
        });
      }
    );
    // });
  }

  ngOnDestroy(): void {
    if (this.subscribeTm) this.subscribeTm.unsubscribe();
  }

  onDiscard() {}

  public async connectToMetamask() {
    // @ts-ignore
    if (typeof window.ethereum === 'undefined' || !window.ethereum) {
      this.snackBar.open(
        'MetaMask plugin is not installed!',
        'Please install MetaMask',
        {
          duration: this.defaultDuration,
        }
      );
      this.isMetamaskConnected = false;
    } else {
      await this.getMetamaskAccounts();
      if (this.accounts == null) return;
      this.isMetamaskConnected = true;
      this.snackBar.open('MetaMask is connected!', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private async getMetamaskAccounts() {
    try {
      // @ts-ignore
      const accounts = await window.ethereum.request({
        method: 'eth_requestAccounts',
      });
      console.log('Accounts: ' + accounts);
      this.accounts = accounts;
    } catch (err) {
      // @ts-ignore
      if (err.code === 4001) {
        // EIP-1193 userRejectedRequest error
        // If this happens, the user rejected the connection request.
        // this.showToast(this.toastService, 'Please connect to MetaMask.', 'warning');
        this.snackBar.open('Please connect to MetaMask.', 'Warn', {
          duration: this.defaultDuration,
        });
      } else {
        /* this.snackBar.open(err + '', 'Error', {
          duration: this.defaultDuration,
        }); */
      }
    }
  }

  async purchase() {
    if (this.useMockup) {
      this.metamaskService
        .submitPurchaseMock(this.accounts[0]) //change to submitPurchase
        .then((result) => {
          console.log('Result', result);
          this.isSubmitted = true;
          this.txHash = result.transactionHash;
          this.snackBar.open(
            `Transaction sent with ID: ${this.txHash}`,
            'Info',
            {
              duration: this.defaultDuration,
            }
          );
          /*   this.snackBar.open(JSON.stringify(result), 'Info', {
            duration: this.defaultDuration,
          }); */
          //remove this
          this.offeringDetailsId = this.createCorrelationId(4);
          const date = new Date().toDateString();
          const message =
            'Message with id ' +
            this.offeringDetailsId +
            ' was created at ' +
            date +
            ' with status ' +
            'PROCESSED';
          this.snackBar.open(message, 'Request status', {
            duration: 6000,
          });
          //remove this
        })
        .catch((error: any) => {
          console.error(error);
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        });
    } else {
      //No mockup
      this.subscribeTm = this.tmApiService
        .purchase('abc')
        .subscribe((response: any) => {
          console.log('Response', response);
          this.snackBar.open(JSON.stringify(response), 'Info', {
            duration: this.defaultDuration,
          });
          if (!response.to || !response.data) {
            this.snackBar.open('No data for blockchain', 'Error', {
              duration: this.defaultDuration,
            });
            return;
          }
          if (!this.isMetamaskConnected) {
            this.snackBar.open('Please connect to MetaMask', 'Error', {
              duration: this.defaultDuration,
            });
            return;
          }
          this.metamaskService
            .submitPurchase(response)
            .then((result: any) => {
              console.log('Result', result);
              this.isSubmitted = true;
              this.txHash = result.hash;
              this.snackBar.open(JSON.stringify(result), 'Info', {
                duration: this.defaultDuration,
              });
            })
            .catch((error: any) => {
              console.error(error);
              this.snackBar.open(JSON.stringify(error), 'Error', {
                duration: this.defaultDuration,
              });
            });
        });
    }
  }

  public priceTo2Decimals(price: any) {
    const val = Math.round(price * 100) / 100;
    return val.toFixed(2);
  }

  private decodeCap(offering: any) {
    if (offering.cap.subscription) {
      this.cap = offering.cap.subscription;
      this.capType = 'subscription';
      switch (offering.cap.subscriptionType) {
        case 'm':
          this.capAdditional = 'minute';
          break;
        case 'H':
          this.capAdditional = 'hour';
          break;
        case 'W':
          this.capAdditional = 'week';
          break;
        case 'M':
          this.capAdditional = 'month';
          break;
        case 'V':
          this.capAdditional = 'year';
          break;
      }
    } else if (offering.cap.downloads) {
      this.cap = offering.cap.downloads;
      this.capType = 'downloads';
    } else if (offering.cap.volume) {
      this.cap = offering.cap.volume;
      this.capType = 'volume';
      this.capAdditional = offering.cap.volumeType;
    }
  }

  private createCorrelationId(parts: number) {
    const stringArr = [];
    for (let i = 0; i < parts; i++) {
      // tslint:disable-next-line:no-bitwise
      const S4 = (((1 + Math.random()) * 0x10000) | 0)
        .toString(16)
        .substring(1);
      stringArr.push(S4);
    }
    return stringArr.join('-');
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    if (!this.useMockup) {
      this.provenanceApiService.checkRequestStatus(this.offering).subscribe(
        (response: any) => {
          if (!response) return;
          console.log('Response', response);

          const message =
            response.message +
            ' was created at ' +
            response.finalized +
            ' with status ' +
            response.status;
          this.snackBar.open(message, 'Request status', {
            duration: 6000,
          });
        },
        (error) => {
          console.log('Error', error);
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        }
      );
    } else {
      setTimeout(() => {
        const date = new Date().toDateString();
        const message =
          'Message with id ' +
          this.offeringDetailsId +
          ' was created at ' +
          date +
          ' with status ' +
          'PROCESSED';
        this.snackBar.open(message, 'Request status', {
          duration: 6000,
        });
      }, 1000);
    }
  }
}
