import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { FormControl, FormGroup, NgForm, Validators } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { PatApiService } from 'src/app/services/api/pat.api.service';
import { ActivatedRoute, Router } from '@angular/router';
import { off } from 'process';
import { Buffer } from 'buffer';
const BASE58 = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';
const basex = require('base-x');
const bs58 = basex(BASE58);

@Component({
  selector: 'pat',
  templateUrl: './pat.component.html',
  styleUrls: ['./pat.component.scss'],
})
export class PatComponent implements OnInit, OnDestroy {
  private accounts: any = null;
  private defaultDuration: number = 3000;
  protected txHash: any;

  questionnaire: any;
  assetId: any;
  offeringId: any;
  patForm: any;
  offering: any;
  correlationId: any;
  subscribePat: any;
  subscribePat2: any;

  constructor(
    private snackBar: MatSnackBar,
    private patApiService: PatApiService,
    private route: ActivatedRoute,
    private provenanceApiService: ProvenanceApiService,
    private router: Router
  ) {}

  ngOnInit() {
    this.route.queryParams.subscribe((params) => {
      if (!params['aid']) this.router.navigate(['/offering-definition']);
      this.offeringId = params['oid'];
      this.assetId = params['aid'];
    });
    if (this.offeringId) {
      this.subscribePat = this.provenanceApiService
        .getOfferingDetails(this.offeringId)
        .subscribe(
          (response) => {
            console.log('Offering details', response);

            this.offering = response;
          },
          (error) => {
            console.log('Error Offering confirmed', error);
          }
        );
    }
    this.callQuestions();
  }

  ngOnDestroy(): void {
    if (this.subscribePat) this.subscribePat.unsubscribe();
    if (this.subscribePat2) this.subscribePat2.unsubscribe();
  }

  onDiscard() {
    this.patForm.reset();
    this.router.navigate(['/offering-definition']);
  }

  onSubmit() {
    if (this.patForm.valid) {
      this.doSubmit();
    } else {
      this.snackBar.open('Please fill all the required fields', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private createQuestionnaireForm() {
    if (this.questionnaire && this.questionnaire.questions) {
      const controls: any = {};
      for (const question of this.questionnaire.questions) {
        if (question.required)
          controls[question.id] = new FormControl('', Validators.required);
        else controls[question.id] = new FormControl('');
      }
      // console.log('controls: ', JSON.stringify(controls));
      this.patForm = new FormGroup(controls);
    }
  }

  private async callQuestionsMock() {
    this.questionnaire = JSON.parse(this.patApiService.retrieveQuestionsMock());
  }

  private async createCorrelationId(parts: number) {
    const stringArr = [];
    for (let i = 0; i < parts; i++) {
      // tslint:disable-next-line:no-bitwise
      const S4 = (((1 + Math.random()) * 0x10000) | 0)
        .toString(16)
        .substring(1);
      stringArr.push(S4);
    }
    return stringArr.join('-');
  }

  /* convertBase58ToJson(base58: any) {
    // Decode the Base58 string back to a JSON string
    const charCodeArray = Array.from(bs58.decode(base58));
    const decodedJSONString = charCodeArray.map(x => String.fromCharCode(x)).join('');

    try {
      return JSON.parse(decodedJSONString);
    } catch (error) {
      console.error('Error decoding Base58 data:', error);
    }

  } */

  convertJsonToBase58(jsonData: any): String {
    const jsonStr = JSON.stringify(jsonData);
    return bs58.encode(Buffer.from(jsonStr, 'utf8'));
  }

  private async callQuestions() {
    this.correlationId = await this.createCorrelationId(4);
    console.log('Correlation ID', this.correlationId);
    const request = {
      // asset: 'DATASET',
      // offering: this.offeringId,
      offering_id: this.correlationId, //correlationId
      dcterms_identifier: this.assetId,
      //TODO Implement the following fields
      dcterms_title: 'Mock Title',
      dcterms_description: 'Mock Description',
      dcterms_type: 'Dataset',
      offering: this.offering || {},
    };
    console.log('Request callQuestions', JSON.stringify(request));
    const codedString = this.convertJsonToBase58(request);
    console.log('Request callQuestions encoded Base58', codedString);
    //Example of input
    /* {
      "offering_id": "ff93c6217a3843ab9a1b0c5d5e7f3b2a",
      "dcterms_identifier": "bf07c2b34b6a42ab9c3b2c4e5f8a7c3d",
      "dcterms_type": "Dataset",
      "dcterms_title": "Global Temperature Data",
      "dcterms_description": "Historical global temperature data from 1880 to present.",
      "offering": {
        "subscription": "2Y",
        "downloads": 1500,
        "volume": "50GB"
      } */
    this.subscribePat2 = this.patApiService
      .retrieveQuestions(codedString)
      .subscribe(
        (response) => {
          console.log('Response callQuestions', JSON.stringify(response));
          if (!response) return;
          // @ts-ignore
          this.questionnaire = response;
          this.createQuestionnaireForm();
        },
        (error) => {
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        }
      );
  }

  private doSubmit() {
    const questionnaireCompiled: any = {};
    questionnaireCompiled.cid = this.questionnaire.cid;
    questionnaireCompiled.answers = [];
    for (const question of this.questionnaire.questions) {
      const value = this.patForm.get(question.id + '').value;
      if (!value) continue;
      const answer = {
        id: question.id,
        value: value,
      };
      questionnaireCompiled.answers.push(answer);
    }
    this.calculatePrice(questionnaireCompiled);
  }

  private calculatePrice(questionnaireCompiled: any) {
    console.log(
      'questionnaireCompiled: ',
      JSON.stringify(questionnaireCompiled)
    );
    questionnaireCompiled.cid = this.correlationId;
    const encodedString = this.convertJsonToBase58(questionnaireCompiled);
    this.patApiService.calculatePrice(encodedString).subscribe(
      (response) => {
        // this.showToast(this.toastService, JSON.stringify(response), 'info');
        this.snackBar.open(JSON.stringify(response), 'Info', {
          duration: this.defaultDuration,
        });
        let num = Math.abs(Math.round(Math.random() * 100));
        // @ts-ignore
        if (response && response.price) {
          // @ts-ignore
          num = +response.price;
        }
        // this.parent.form.controls.price.setValue(num);
        this.patForm.reset();
        //TODO Implement navigation back passing the price
        this.router.navigate(['/offering-definition'], {
          queryParams: { price: num },
        });
      },
      (error) => {
        console.log('Error', error);
        this.snackBar.open(JSON.stringify(error), 'Error', {
          duration: this.defaultDuration,
        });
        let num = Math.abs(Math.round(Math.random() * 100));
        this.router.navigate(['/offering-definition'], {
          queryParams: { price: num },
        });
      }
    );
  }

  private calculatePriceMock(questionnaireCompiled: any) {
    //this.showToast(this.toastService, JSON.stringify(questionnaireCompiled), 'info');
    this.snackBar.open(JSON.stringify(questionnaireCompiled), 'Info', {
      duration: this.defaultDuration,
    });
    const num = Math.abs(Math.round(Math.random() * 100));
    this.patForm.reset();
    //TODO Implement navigation back passing the price
  }

  back() {
    this.router.navigate(['/offering-definition']);
  }
}
