import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { Observable } from 'rxjs';
import { environment } from 'src/environments/environment';

@Injectable({
  providedIn: 'root',
})
export class AuthService {
  private authAPIUrl = environment.authAPIUrl + 'api/user-validations';
  private verifyTokenAPI = environment.authAPIUrl + 'api/verify-token'; // (1) ADDED

  private jwtTokenKey = 'jwt'; // sessionStorage key

  constructor(private httpClient: HttpClient) {}

  /**
   * Checks whether the user is logged in.
   * For this example, we consider the user logged in if a JWT token exists in localStorage.
   */
  isLoggedIn(): boolean {
    // Check for the token (adjust the key if necessary)
    return !!localStorage.getItem(this.jwtTokenKey);
  }

  // Generate a new JWT token from the backend using a verifiable presentation token
  public generateJWTToken(vptoken: any): Observable<any> {
    const requestBody = { vp_token: vptoken };
    return this.httpClient.post(this.authAPIUrl, requestBody);
  }

  // ADDED: Verify the given JWT token with the backend
  public verifyToken(jwtToken: string): Observable<any> {
    const requestBody = { jwtToken: jwtToken };
    return this.httpClient.post(this.verifyTokenAPI, requestBody);
  }

  // Store the JWT token in sessionStorage
  setToken(token: string): void {
    localStorage.setItem(this.jwtTokenKey, token);
    // use access jwt token for subdomain
    this.setJwtCookie(token);
  }

  // access jwt token for subdomain
  private setJwtCookie(token: string): void {
    const domain = '.fame-horizon.eu';
    const path = '/';
    const sameSite = 'None';
    const expires = new Date();
    expires.setDate(expires.getDate() + 7);
    document.cookie = `jwt_token=${token};expires=${expires.toUTCString()};path=${path};domain=${domain};Secure;SameSite=${sameSite}`;
  }

  // remove jwt token for subdomain
  private removeJwtCookie(): void {
    const domain = '.fame-horizon.eu';
    const path   = '/';
    const sameSite = 'None';
    document.cookie = `jwt_token=; Max-Age=0; path=${path}; domain=${domain}; Secure; SameSite=${sameSite}`;
  }

  // Retrieve the JWT token from sessionStorage
  getToken(): string | null {
    return localStorage.getItem(this.jwtTokenKey);
  }

  // Clear the JWT token from sessionStorage
  clearToken(): void {
    localStorage.removeItem(this.jwtTokenKey);
    this.removeJwtCookie();
  }

  // Safely decode the JWT to retrieve user details.
  getUserDetails(): any {
    const token = this.getToken();
    if (!token) return null;

    // Must have "header.payload.signature"
    const parts = token.split('.');
    if (parts.length !== 3) {
      console.warn('Invalid JWT format:', token);
      return null;
    }

    try {
      const base64Url = parts[1];
      const base64 = base64Url.replace(/-/g, '+').replace(/_/g, '/');
      const jsonPayload = decodeURIComponent(
        atob(base64)
          .split('')
          .map(c => '%' + ('00' + c.charCodeAt(0).toString(16)).slice(-2))
          .join('')
      );
      return JSON.parse(jsonPayload);
    } catch (error) {
      console.error('Error decoding JWT payload:', error);
      return null;
    }
  }

  // Extract nickname and region from the token's user details
  getNickNameAndRegion(): { nickname: string | null; region: string | null } {
    const userDetails = this.getUserDetails();
    // console.log(userDetails);
    if (userDetails?.nickname) {
      const { nickname, region } = userDetails;
      return { nickname, region };
    }
    return { nickname: null, region: null };
  }

  isTokenExpired(): boolean {
    const userDetails = this.getUserDetails();
    if (!userDetails?.exp) {
      return true; // missing token or exp => treat as expired
    }
    const now = Math.floor(Date.now() / 1000);
    return userDetails.exp < now; // is exp < current time => expired
  }
}