import { Component, OnInit, OnDestroy } from '@angular/core';
import {
  CreateElementArgs,
  QRType,
  URIData,
  ValueResult,
} from '@sphereon/ssi-sdk.qr-code-generator';
import {
  AuthorizationResponsePayload,
  AuthStatusResponse,
  GenerateAuthRequestURIResponse,
} from '@sphereon/ssi-sdk.siopv2-oid4vp-common';
import agent from './agent';
import { environment } from 'src/environments/environment';
import { CommonService } from 'src/app/services/common.service';

@Component({
  selector: 'app-authentication',
  templateUrl: './authentication.component.html',
  styleUrls: ['./authentication.component.scss'],
})
export class AuthenticationComponent implements OnInit, OnDestroy {
  public authRequestURIResponse?: GenerateAuthRequestURIResponse;
  public qrCode?: any;
  public qrProps?: any;
  private registerStateSent: boolean = false;
  private refreshTimerHandle?: any;
  private qrExpirationMs: number = 0;
  private timedOutRequestMappings: Set<GenerateAuthRequestURIResponse> =
    new Set<GenerateAuthRequestURIResponse>();
  private isMounted: boolean = false;
  private readonly definitionId = environment.presentationDefId || 'sphereon';

  constructor(private commonService: CommonService) {}

  ngOnInit() {
    this.qrExpirationMs = (environment.qrCodeExpiresAfterSec ?? 1200) * 1000;
    if (!this.authRequestURIResponse || !this.qrCode) {
      this.generateNewQRCode();
      this.refreshTimerHandle = setTimeout(
        () => this.refreshQRCode(),
        this.qrExpirationMs
      );
    }
    this.isMounted = true;
  }

  ngOnDestroy() {
    if (this.refreshTimerHandle) {
      clearTimeout(this.refreshTimerHandle);
    }
    this.isMounted = false;
  }

  private generateNewQRCode() {
    this.generateAuthRequestURI()
      .then((authRequestURIResponse) => {
        agent
          .qrURIElement(this.createQRCodeElement(authRequestURIResponse))
          .then((qrCode) => {
            const qrCodeHtml = qrCode as any;
            this.registerState(authRequestURIResponse, qrCode);
          });
      })
      .catch((e) => console.error(e));
  }

  private createQRCodeElement(
    authRequestURIResponse: GenerateAuthRequestURIResponse
  ): CreateElementArgs<QRType.URI, URIData> {
    const qrProps: CreateElementArgs<QRType.URI, URIData> = {
      data: {
        type: QRType.URI,
        object: authRequestURIResponse.authRequestURI,
        id: authRequestURIResponse.correlationId,
      },
      onGenerate: (result: ValueResult<QRType.URI, URIData>) => {},
      renderingProps: {
        bgColor: 'white',
        fgColor: '#352575',
        level: 'L',
        size: 200,
        title: 'Sign in',
      },
    };
    this.qrProps = qrProps;
    //console.log(qrProps);
    return qrProps;
  }

  private async generateAuthRequestURI(): Promise<GenerateAuthRequestURIResponse> {
    return await agent.siopClientCreateAuthRequest({
      definitionId: this.definitionId,
    });
  }

  private refreshQRCode() {
    console.log('Timeout qrDataexpired, refreshing QR code...');
    if (this.qrExpirationMs > 0) {
      if (this.authRequestURIResponse) {
        this.timedOutRequestMappings.add(this.authRequestURIResponse);
      }
      this.registerStateSent = false;
      this.generateNewQRCode();
    }
  }

  private registerState(
    authRequestURIResponse: GenerateAuthRequestURIResponse,
    qrCode: JSX.Element
  ) {
    if (
      this.authRequestURIResponse?.correlationId ===
      authRequestURIResponse.correlationId
    ) {
      return;
    }
    this.authRequestURIResponse = authRequestURIResponse;
    this.qrCode = authRequestURIResponse;
    this.pollAuthStatus(authRequestURIResponse);
  }

  private pollAuthStatus(
    authRequestURIResponse: GenerateAuthRequestURIResponse
  ) {
    let interval = setInterval(async () => {
      if (!this.qrCode) {
        clearInterval(interval);
      } else {
        let authStatus: AuthStatusResponse =
          await agent.siopClientGetAuthStatus({
            correlationId: authRequestURIResponse.correlationId,
            definitionId: authRequestURIResponse.definitionId,
          });

        if (authStatus.status === 'sent') {
        } else if (authStatus.status === 'verified') {
          const payload = authStatus.payload;
          this.commonService.refinedEvent.emit(payload);
          clearInterval(interval);
        } else if (authStatus.status === 'error') {
          clearInterval(interval);
          return Promise.reject(authStatus.error);
        }

        if (this.timedOutRequestMappings.has(authRequestURIResponse)) {
          try {
            console.log('Cancelling timed out auth request.');
            await agent['siopClientRemoveAuthRequestSession']({
              correlationId: authRequestURIResponse.correlationId,
              definitionId: authRequestURIResponse.definitionId,
            });
            this.timedOutRequestMappings.delete(authRequestURIResponse);
            clearInterval(interval);
          } catch (error) {
            console.log(error);
          }
        }
      }
    }, 1000);
  }
}