import { Component, inject } from '@angular/core';
import { CommonModule } from '@angular/common';
import { IframeComponent, IframeProps } from '../iframe/iframe.component';
import { ActivatedRoute, Router } from '@angular/router';
import { MatProgressSpinnerModule } from '@angular/material/progress-spinner';
import { SnackbarService } from '../../services/snackbar/snackbar.service';
import { PolicyIntegrationService } from '../../services/policy-integration.service';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { MatDialog, MatDialogModule } from '@angular/material/dialog';
import { APIResponse, OpenAPIService } from '../../services/open-api.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { UnpublishAssetConfirmationDialogComponent } from '../unpublish-asset-confirmation-dialog/unpublish-asset-confirmation-dialog.component';
import { firstValueFrom, lastValueFrom, Observer } from 'rxjs';
import { environment } from '../../../environments/environment';
import { AssetLineageTracing } from 'src/app/services/api/dtos/asset.dto';
import { AuthService } from 'src/app/auth/auth.service';

import * as dayjs from 'dayjs'

@Component({
  selector: 'component-details',
  standalone: true,
  imports: [CommonModule, IframeComponent, MatProgressSpinnerModule, MatIconModule, MatButtonModule, MatDialogModule],
  template: `
    <div *ngIf="loading" class="loading-container">
      <mat-spinner></mat-spinner>
    </div>

    <div

      *ngIf="!loading && iframeProps"
      class="iframe-container"

    >
      
    <div *ngIf="!!authService.getToken()" style="display:flex;justify-content:center;margin-bottom:1rem;">
       <button [disabled]="checkButtonDisabled" mat-raised-button color="primary" (click)="integrityCheck()">
        <mat-icon>verified</mat-icon>
        Check asset integrity
      </button>
    </div>
     
      <p *ngIf="lineageElements && lineageElements.length >0" class="info-label">
        <span>Previous versions</span>
         <a style="margin-right:1rem" *ngFor="let item of lineageElements; index as i" [href]="'/asset/' + item.value?._id" target="_blank">{{formatDate(item.value?.updatedAt)}}</a>
      </p>

     
      <iframe-component
        [iframeProps]="iframeProps"
        (loaded)="onIframeLoad($event)"
      ></iframe-component>
      <p *ngIf="contentDeliveryServiceURL" class="info-label">
        <span>Content Delivery Service</span><a target="_blank" [href]="contentDeliveryServiceURL">{{contentDeliveryServiceURL}}</a>
      </p>
      <p *ngIf="showAccessType" class="info-label">
        <span>Access Type</span>{{accessType}}
      </p>

      <!-- Unpublished Asset Warning Banner -->
      <div class="card warning-card" *ngIf="isArchived">
        <div class="warning-content">
          <mat-icon class="warning-icon">warning</mat-icon>
          <div class="warning-text">
            <h3>This asset has been unpublished</h3>
            <p>This asset is no longer available. It is displayed here for reference purposes only. This status cannot be reversed.</p>
          </div>
        </div>
      </div>

      <div *ngIf="isWriteable && addNewOfferingLink && editPolicyVisible" class="component-details-buttons">
        <div class="button-wrapper">
          <div class="tooltip">
            <a mat-raised-button color="primary" [href]="addNewOfferingLink" >
            <mat-icon>add_circle_outline</mat-icon>
              Add Offering
            </a>
            <span class="tooltiptext">Create a new commercial offering for this asset</span>
          </div>
          <div class="tooltip">
            <a mat-raised-button color="accent" [href]="this.getEditPolicyLink()" target="_blank">
              <mat-icon>edit</mat-icon>
              Edit Policy
            </a>
            <span class="tooltiptext">Modify access policies and sharing rules for this asset</span>
          </div>
        </div>
        <div class="button-wrapper">
          <div class="tooltip">
            <button mat-raised-button (click)="onRepublishAsset()">
              <mat-icon>refresh</mat-icon>
              Republish
            </button>
            <span class="tooltiptext">Create a new version of this asset with updated information</span>
          </div>
          <div class="tooltip">
            <button mat-raised-button color="warn" (click)="onUnpublishAsset()">
              <mat-icon>delete_forever</mat-icon>
              Unpublish
            </button>
            <span class="tooltiptext">Remove this asset from the catalogue (cannot be undone)</span>
          </div>
        </div>
      </div>
      <br><br>
    </div>

    <!-- Loading Overlay for unpublish operation -->
    <div class="spinner-overlay" *ngIf="isLoading">
      <mat-progress-spinner mode="indeterminate" diameter="60"></mat-progress-spinner>
    </div>
  `,
  styles: [
    `
      .loading-container {
        display: flex;
        justify-content: center;
        align-items: center;
        height: 100vh; /* Adjust as necessary */
      }

      .snackbar-success {
        background-color: #4caf50; /* Green */
        color: #fff;
      }

      .snackbar-error {
        background-color: #f44336; /* Red */
        color: #fff;
      }

      .iframe-container {
        max-width: 1100px;
        margin: 0 auto;
        // height: 150vh; /* default for small/medium screens */
      }
      .info-label {
          border: 1px solid var(--gray-400);
          padding: 0.5rem 1rem;
          border-radius: 8px;
          span {
              font-weight: bold;
              color: var(--gray-800);
              margin-right:1.5rem
          }
      }
      .component-details-buttons {
        display: flex;
        flex-direction: column;
        align-items: center;
        margin-top: 1rem;
        gap: 1rem;

        .button-wrapper {
          display: flex;
          gap: 1rem; // space between buttons
        }
      
        a[mat-raised-button], button[mat-raised-button] {
          background-color: #334f65 !important;
          color: white;
          font-weight: 500;
          text-transform: none;

          display: flex;
          align-items: center;
          justify-content: center;
          gap: 0.5rem;

          // Set consistent width to accommodate all button labels
          min-width: 200px;
          width: 200px;
          padding: 8px 16px;

          &:hover {
            background-color: #40617a !important;
          }

          mat-icon {
            font-size: 20px;
          }
        }

        button[mat-raised-button][color="warn"] {
          background-color: #dc3545 !important;

          &:hover {
            background-color: #c82333 !important;
          }
        }
      }

      .spinner-overlay {
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(255, 255, 255, 0.8);
        display: flex;
        justify-content: center;
        align-items: center;
        z-index: 9999;
      }

      /* Warning Card for Unpublished Assets */
      .warning-card {
        background: linear-gradient(135deg, #fff5f5 0%, #fee2e2 100%);
        border-left: 6px solid #ef4444;
        box-shadow: 0 4px 12px rgba(239, 68, 68, 0.1);
        border-radius: 8px;
        margin: 16px 0;

        .warning-content {
          display: flex;
          align-items: flex-start;
          gap: 16px;
          padding: 20px;

          .warning-icon {
            color: #ef4444;
            font-size: 2rem;
            width: 2rem;
            height: 2rem;
            flex-shrink: 0;
            margin-top: 2px;
          }

          .warning-text {
            flex: 1;

            h3 {
              color: #dc2626;
              font-size: 1.3rem;
              font-weight: 600;
              margin: 0 0 8px 0;
            }

            p {
              color: #991b1b;
              font-size: 1rem;
              line-height: 1.5;
              margin: 0;
            }
          }
        }
      }

      /* Tooltip styling */
      .tooltip {
        position: relative;
        display: inline-block;
      }

      .tooltip .tooltiptext {
        visibility: hidden;
        width: max-content;
        max-width: 200px;
        background-color: #000;
        color: #fff;
        text-align: center;
        padding: 5px 8px;
        border-radius: 6px;
        position: absolute;
        bottom: 100%;
        left: 50%;
        transform: translateX(-50%);
        margin-bottom: 8px;
        z-index: 10;
        opacity: 0;
        transition: opacity 0.2s ease-in-out;
        pointer-events: none;
      }

      .tooltip:hover .tooltiptext {
        visibility: visible;
        opacity: 1;
      }
    `,
  ],
})

export class ComponentDetails {
  iframeProps: IframeProps | undefined;
  policyIntegrationService: PolicyIntegrationService = inject(PolicyIntegrationService);
  openAPIService: OpenAPIService = inject(OpenAPIService);
  route: ActivatedRoute = inject(ActivatedRoute);
  id: string = this.route.snapshot.params['id']
  isWriteable: boolean | null = null;
  isOwner: boolean = false;
  isArchived: boolean = false;
  accessType: string | null = null
  contentDeliveryServiceURL: string | null = null
  loading: boolean = true;
  checkButtonDisabled: boolean = false;
  isLoading: boolean = false; // For unpublish operation spinner
  addNewOfferingLink: string | null = null;
  lineageElements: APIResponse[] | null = null
  private snackbarService: SnackbarService = inject(SnackbarService);
  private dialog: MatDialog = inject(MatDialog);
  private provenanceApiService: ProvenanceApiService = inject(ProvenanceApiService);
  authService: AuthService = inject(AuthService);
  editPolicyVisible: boolean = false;

  showAccessType = false;

  getEditPolicyLink() {
    return this.policyIntegrationService.getEditPolicyLink(this.id)
  }

  constructor(private router: Router) { }

  formatDate(date: string): string {
    return dayjs(date).format("DD-MM-YYYY HH:mm")
  }

  setContentDeliveryServiceURL(element: any): void {
    const accessPlatformsPredicateId = "68f235d53fc5c92a2dfb8da2"
    const reference = element?.value?.reference || []
    this.contentDeliveryServiceURL = reference.find((el: any) => el.predicate === accessPlatformsPredicateId)?.data?.[0]?.url || null
  }

  async integrityCheck() {
    this.checkButtonDisabled = true
    const value = await lastValueFrom(this.provenanceApiService.getAssetLineage(this.id));
    const trustworthy = value.trustworthy
    const panelClass = trustworthy ? "snackbar-success" : "snackbar-error"
    const msg = `Asset is ${trustworthy ? "" : "not"} trustworthy`
    this.snackbarService.openSnackBar(msg, "", { verticalPosition: "top", duration: 5000, panelClass })
    this.checkButtonDisabled = false

  }

  async getAssetLineage() {
    const value = await lastValueFrom(this.provenanceApiService.getAssetLineage(this.id));

    const { lineage } = value
    // const lineage = "Ay3vb2Gtb5fMoAB8P"
    if (lineage) {

      const ids = lineage.split(",")
      this.lineageElements = (await Promise.all(
        ids.map(id => this.openAPIService.getAssetDetails(id, ["updatedAt"]))
      )).filter(e => e.status === 200)
    }

  }

  async ngOnInit(): Promise<void> {
    this.showSpinner();

    // Step 1: Check if user has visibility to this asset
    const visibilityResponse = await this.policyIntegrationService.assetMarketPlaceVisibilityController.checkOne(this.id);

    // If user doesn't have visibility, redirect to home page
    if (!visibilityResponse?.json?.hasVisibility) {
      console.log('User does not have visibility to this asset, redirecting to home');
      this.router.navigate(['/']);
      return;
    }

    // Step 2: Generate iframe URL and check ownership in parallel
    const [
      ownershipResponse,
      { value: url, status }
    ] = await Promise.all([
      // Check if the user is owner of the asset
      this.policyIntegrationService.assetContentAccessController.checkOne(this.id),
      // Generates the URL for the iframe with the component details page
      this.openAPIService.generateAssetDetailsIframeURL(this.id)
    ]);

    // Handle authentication/authorization errors from iframe URL generation
    if (status === 401) {
      this.router.navigate(["/login"]);
      return;
    }
    if (status === 403) {
      this.router.navigate(["/"]);
      return;
    }
    if (status === 200) {
      this.iframeProps = { url };
    }

    // Extract ownership and access type info
    const ownershipJson = ownershipResponse?.json;
    this.accessType = ownershipJson?.accessType || null;
    this.isOwner = ownershipJson?.hasAccess === true;
    this.isArchived = ownershipJson?.isActive === false;
    this.isWriteable = this.isOwner && !this.isArchived;

    if (this.accessType) {
      setTimeout(() => {
        this.showAccessType = true;
      }, 10000);
    }

    // Obtains the asset name to build the add new offering link
    const element = await this.openAPIService.getAssetDetails(this.id, ["reference"])
    this.setContentDeliveryServiceURL(element)
    const elementName = element?.value?.name


    this.addNewOfferingLink = this.policyIntegrationService.getAddNewOfferingLink(this.id, elementName)

    this.getAssetLineage()
  }

  showSpinner(): void {
    setTimeout(() => {
      this.loading = false;
    }, 4000);
  }

  onIframeLoad(success: boolean): void {
    if (success) {
      // this.snackbarService.success('Content loaded successfully!', 'Close', 3000);
      // Show button after 5 seconds
      setTimeout(() => {
        this.editPolicyVisible = true;
      }, 5000);
    } else {
      this.snackbarService.error('Failed to load content. Please try again.', 'Close', 3000);
    }
  }

  /**
   * Navigate to asset editing form
   */
  onRepublishAsset(): void {
    this.router.navigate(['/asset-publishment', this.id]);
  }

  /**
   * Show unpublish confirmation dialog and handle unpublish process
   */
  async onUnpublishAsset(): Promise<void> {
    const dialogRef = this.dialog.open(UnpublishAssetConfirmationDialogComponent, {
      width: '600px',
      data: {
        assetId: this.id
      }
    });

    const confirmed = await firstValueFrom(dialogRef.afterClosed());
    if (!confirmed) {
      return; // User cancelled
    }

    // Show loading spinner
    this.isLoading = true;

    try {
      await firstValueFrom(this.provenanceApiService.unpublishAsset(this.id));
      this.snackbarService.success('Asset successfully unpublished.', 'Close', 3000);

      // Reload the current page to reflect the unpublished status
      window.location.reload();
    } catch (error: any) {
      console.error('Error unpublishing asset:', error);

      if (error.status === 404) {
        this.snackbarService.error('Asset not found.', 'Close', 5000);
      } else if (error.status === 401 || error.status === 403) {
        this.snackbarService.error('You do not have permission to unpublish this asset.', 'Close', 5000);
      } else {
        this.snackbarService.error('Error unpublishing asset. Please try again.', 'Close', 5000);
      }
    } finally {
      // Hide loading spinner
      this.isLoading = false;
    }
  }
}