import { Component, ElementRef, EventEmitter, Input, Output, ViewChild } from '@angular/core';
import { CommonModule } from '@angular/common';
import { DomSanitizer, SafeResourceUrl } from '@angular/platform-browser';
import { Loading } from './loading.component';
import { AuthService } from 'src/app/auth/auth.service';

export interface IframeProps {
    url?: string
}

export interface IframeRequest {
    "id": string,
    "json": any
}

@Component({
    selector: 'iframe-component',
    standalone: true,
    imports: [CommonModule, Loading],
    template: `
        <section class="container">
            <iframe #contentChild [ngStyle]="{ 'width':'100%', 'border':0,'height.px': this.height}" [src]="sanitizedUrl" border="0" (load)="onLoad()"
            (error)="onError()"></iframe>
        </section>
    `,
    styleUrls: [],
})

export class IframeComponent {
    @Input() iframeProps!: IframeProps;
    sanitizedUrl: SafeResourceUrl = this.getSanitizedURL("");
    height: number = 0

    dom: HTMLIFrameElement | null = null
    @ViewChild("contentChild")
    set appShark(el: ElementRef<HTMLIFrameElement>) {

        this.dom = el.nativeElement
    };
    getSanitizedURL(url: string): SafeResourceUrl {
        return this.domSanitizer.bypassSecurityTrustResourceUrl(url);
    }

    constructor(private domSanitizer: DomSanitizer, private authService: AuthService) {
        this.processMessage = this.processMessage.bind(this)
        //@ts-ignore
        window.iframe = this
    }

    sendMessage(msg: any) {
        if (!this.dom) return
        this.dom.contentWindow?.postMessage(msg, "*")
    }

    processMessage(event: MessageEvent) {
        const data = event.data
        if (data.action === "resize") {
            this.sendMessage({ action: "metadata", value: { jwtToken: this.authService.getToken() } })
            this.height = data.height
        }
    }

    async getIframeURL() {
        return this.iframeProps.url
    }

    async ngOnInit() {
        const url = await this.getIframeURL()
        if (!url) return
        this.sanitizedUrl = this.getSanitizedURL(url)
        window.addEventListener("message", this.processMessage, false)
    }

    ngOnDestroy() {
        window.removeEventListener("message", this.processMessage)
    }

    @Output() loaded = new EventEmitter<boolean>();
    onLoad() {
        this.loaded.emit(true); // Emit success
    }

    onError() {
        this.loaded.emit(false); // Emit failure
    }
}