import { Component, Inject } from '@angular/core';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';

export interface MarkdownPreviewModalData {
  html: string;
  title?: string;
}

@Component({
  selector: 'app-markdown-preview-modal',
  template: `
    <div class="markdown-preview-modal">
      <div class="modal-header">
        <h2 mat-dialog-title>{{ data.title || 'Markdown Preview' }}</h2>
        <button
          mat-icon-button
          class="close-button"
          (click)="onClose()"
          [attr.aria-label]="'Close dialog'"
        >
          <mat-icon>close</mat-icon>
        </button>
      </div>

      <mat-dialog-content class="modal-content">
        <div
          class="preview-content"
          [innerHTML]="data.html"
        ></div>
      </mat-dialog-content>
    </div>
  `,
  styles: [`
    .markdown-preview-modal {
      .modal-header {
        display: flex;
        justify-content: space-between;
        align-items: center;
        padding-bottom: 16px;
        border-bottom: 1px solid #e0e0e0;
        margin-bottom: 16px;

        h2 {
          margin: 0;
          color: #334f65;
          font-size: 1.5rem;
          font-weight: 600;
        }

        .close-button {
          color: #666;

          &:hover {
            color: #334f65;
            background-color: rgba(51, 79, 101, 0.1);
          }
        }
      }

      .modal-content {
        min-height: 200px;
        max-height: 60vh;
        overflow-y: auto;
        padding: 0;

        .preview-content {
          line-height: 1.6;
          color: #333;
          padding: 16px 20px;

          // Style the rendered markdown content
          h1, h2, h3, h4, h5, h6 {
            color: #334f65;
            margin-top: 1.5em;
            margin-bottom: 0.5em;

            &:first-child {
              margin-top: 0;
            }
          }

          h1 { font-size: 2rem; }
          h2 { font-size: 1.75rem; }
          h3 { font-size: 1.5rem; }
          h4 { font-size: 1.25rem; }
          h5 { font-size: 1.1rem; }
          h6 { font-size: 1rem; }

          p {
            margin-bottom: 1em;

            &:last-child {
              margin-bottom: 0;
            }
          }

          strong {
            font-weight: 600;
            color: #334f65;
          }

          em {
            font-style: italic;
          }

          ul, ol {
            margin: 1em 0;
            padding-left: 2em;

            li {
              margin-bottom: 0.5em;

              &:last-child {
                margin-bottom: 0;
              }
            }
          }

          ul {
            list-style-type: disc;
          }

          ol {
            list-style-type: decimal;
          }

          br {
            line-height: 0.5;
          }
        }
      }
    }

    // Responsive design
    @media (max-width: 768px) {
      .markdown-preview-modal .modal-content {
        max-height: 50vh;
      }
    }
  `]
})
export class MarkdownPreviewModalComponent {

  constructor(
    public dialogRef: MatDialogRef<MarkdownPreviewModalComponent>,
    @Inject(MAT_DIALOG_DATA) public data: MarkdownPreviewModalData
  ) {
    // Enable ESC key to close the dialog
    this.dialogRef.keydownEvents().subscribe(event => {
      if (event.key === 'Escape') {
        this.onClose();
      }
    });
  }

  onClose(): void {
    this.dialogRef.close();
  }
}