import { Component, OnInit, EventEmitter, Output, HostListener } from '@angular/core';
import { Router } from '@angular/router';
import { AuthService } from 'src/app/auth/auth.service';
import { MatButtonModule } from '@angular/material/button';
import { MatDividerModule } from '@angular/material/divider';
import { MatIconModule } from '@angular/material/icon';
import { CommonModule } from '@angular/common';
import { CommonService } from 'src/app/services/common.service';
import { SnackbarService } from 'src/app/services/snackbar/snackbar.service';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatInputModule } from '@angular/material/input';
import { FormsModule } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';

@Component({
  selector: 'app-navigation-bar',
  templateUrl: './navigation-bar.component.html',
  styleUrls: ['./navigation-bar.component.scss'],
  standalone: true,
  imports: [MatButtonModule, MatDividerModule, MatIconModule, CommonModule, FormsModule, MatFormFieldModule, MatInputModule],
})
export class NavigationBarComponent implements OnInit {
  menuOpen: boolean = false;
  userLogin: any;
  buttonState: boolean = true;
  logoState: boolean = true;

  @Output() searchNavigation = new EventEmitter<void>();

  // Flags to handle scroll-based behavior
  isScrolled = false;
  // For the search input
  searchQuery = '';

  constructor(
    private commonService: CommonService,
    private authService: AuthService,
    private router: Router,
    private snackbarService: SnackbarService,
    private snackBar: MatSnackBar
  ) {
    this.commonService.popCloseEvent.subscribe((value) => {
      this.userLogin = value?.firstName + value?.lastName;
      this.buttonState = true;
      this.logoState = false;
    });
  }

  // 1) Listen to window scroll
  @HostListener('window:scroll', [])
  onWindowScroll() {
    // If scrolled more than 50px from the top, set isScrolled = true
    this.isScrolled = window.pageYOffset > 50;
  }

  ngOnInit() {
    this.checkLoginStatus();
  }

  checkLoginStatus() {
    // If no token or it's expired => show "Log in" button
    if (!this.authService.getToken() || this.authService.isTokenExpired()) {
      this.buttonState = false; // "Log In"
      this.logoState = true;
      this.authService.clearToken(); // optional: remove expired token
    } else {
      this.buttonState = true;  // "Log Out"
      this.logoState = false;
    }
  }

  redirectToLogin() {
    this.router.navigate(['/login']);
    this.closeMenu();
  }

  SignOut() {
    try {
      this.authService.clearToken();
      this.logoState = true;
      this.buttonState = false;
      this.checkLoginStatus(); // Update button state
      this.router.navigate(['/']);
      // this.snackbarService.success('Successfully signed out!', 'Close'); // Show success snackbar
      this.closeMenu();
    } catch (error) {
      this.snackbarService.error('Error during sign-out. Please try again.', 'Close');
    }
  }

  navigateTo(page: string) {
    const publicPages = ['/about-us', '/helpdesk', '/', '/search'];
    if (this.buttonState || publicPages.includes(page)) {
      if (page === '/search') {
        this.searchNavigation.emit(); // Emit the event when navigating to search
      }
      this.router.navigate([page]);
    } else {
      this.redirectToLogin();
    }
    this.closeMenu();
  }

  toggleMenu() {
    this.menuOpen = !this.menuOpen;
  }

  closeMenu() {
    this.menuOpen = false;
  }

  // 5) Searching
  // onKeyDown(event: KeyboardEvent) {
  //   if (event.key === 'Enter') {
  //     this.performSearch();
  //   }
  // }

  performSearch() {
    // Example search logic
    console.log('Searching for:', this.searchQuery);
    // Possibly navigate to a search results page
    // this.router.navigate(['/search-results'], { queryParams: { q: this.searchQuery } });
  }

  goToAdvancedSearch() {
    this.router.navigate(['/fdac']);
  }
  
  onSearch() {
    if (!this.searchQuery || this.searchQuery.trim() === '') {
      // Show a snack bar message if the search field is empty
      this.snackBar.open('Please enter a search query.', 'Close', { duration: 3000 });
      return;  // Stop the navigation if the search field is empty
    }
    // Always navigate to /search with the search query
    this.router.navigate(['/fdac'], { queryParams: { query: this.searchQuery } });
  }

  onFocus(): void {
    // this.stopCycle();
  }
  
  /**
   * If the user leaves (blur) and the field is empty, restart the animation cycle.
   */
  onBlur(): void {
    if (!this.searchQuery) {
      // this.startCycle();
    }
  }

  onKeyDown(event: KeyboardEvent) {
    if (event.key === 'Enter') {
      this.onSearch();
    }
  }
}