import { Component, ViewChild, AfterViewInit, OnInit, HostListener } from '@angular/core';
import { BreakpointObserver } from '@angular/cdk/layout';
import { MatSidenav } from '@angular/material/sidenav';
import { NavigationEnd, Router } from '@angular/router';
import { filter } from 'rxjs/operators';
import { UntilDestroy, untilDestroyed } from '@ngneat/until-destroy';
import { AuthService } from 'src/app/auth/auth.service';
import { ChangeDetectorRef } from '@angular/core';
import { SnackbarService } from 'src/app/services/snackbar/snackbar.service';

@UntilDestroy()
@Component({
  selector: 'app-responsive-sidenav-end-user',
  templateUrl: './responsive-sidenav-end-user.component.html',
  styleUrls: ['./responsive-sidenav-end-user.component.scss'],
})
export class ResponsiveSidenavEndUserComponent implements AfterViewInit, OnInit {
  @ViewChild(MatSidenav) sidenav!: MatSidenav;

  // Add this to fix the "isMobile does not exist" error
  public isMobile: boolean = false;

  menuOpen: boolean = false;
  userLogin: any;
  buttonState: boolean = false;
  logoState: boolean = true;
  nickName: string = '';  // To store the nickname from the JWT
  region: string = '';    // To store the region from the JWT
  activeLink: string = '/';

  isScrolled: boolean = false;
  searchQuery: string = '';

  @HostListener('window:scroll', [])
  onWindowScroll(): void {
    // Set a threshold for when to show the search input (e.g., 100px scroll)
    this.isScrolled = window.pageYOffset > 100;
  }

  // goToAdvancedSearch() {
  //   this.router.navigate(['/search']);
  // }

  // 5) Searching
  // onKeyDown(event: KeyboardEvent) {
  //   if (event.key === 'Enter') {
  //     this.performSearch();
  //   }
  // }

  performSearch() {
    // Example search logic
    console.log('Searching for:', this.searchQuery);
    // Possibly navigate to a search results page
    // this.router.navigate(['/search-results'], { queryParams: { q: this.searchQuery } });
  }

  constructor(
    private observer: BreakpointObserver,
    private router: Router,
    private authService: AuthService,
    private cdRef: ChangeDetectorRef,
    private snackbarService: SnackbarService
  ) {
    // Listen to route changes to update the active link
    this.router.events.subscribe(event => {
      if (event instanceof NavigationEnd) {
        this.activeLink = this.router.url;
      }
    });
  }

  ngOnInit() {
    this.checkLoginStatus();
    this.loadUserDetails();
  }

  ngAfterViewInit() {
    // Observe screen size and update isMobile accordingly
    this.observer
      .observe(['(max-width: 800px)'])
      .pipe(untilDestroyed(this))
      .subscribe((res) => {
        if (res.matches) {
          // Mobile
          this.isMobile = true;
          // If using [mode]="(isMobile ? 'over' : 'side')" in HTML:
          this.sidenav.mode = 'over';
          this.sidenav.close();
        } else {
          // Desktop
          this.isMobile = false;
          this.sidenav.mode = 'side';
          this.sidenav.open();
        }
        // Trigger change detection
        this.cdRef.detectChanges();
      });

    // Close sidenav after route changes if it’s mobile
    this.router.events
      .pipe(
        untilDestroyed(this),
        filter((e) => e instanceof NavigationEnd)
      )
      .subscribe(() => {
        if (this.isMobile && this.sidenav.opened) {
          this.sidenav.close();
        }
      });

    // Avoid ExpressionChangedAfterItHasBeenCheckedError
    this.cdRef.detectChanges();
  }

  checkLoginStatus() {
    // If no token or it's expired => show "Log in" button
    if (!this.authService.getToken() || this.authService.isTokenExpired()) {
      this.buttonState = false; // "Log In"
      this.logoState = true;
      this.authService.clearToken(); // optional: remove expired token
    } else {
      this.buttonState = true;  // "Log Out"
      this.logoState = false;
    }
  }

  loadUserDetails() {
    const token = this.authService.getToken();
    if (!token) {
      // no token => user not logged in => skip
      return;
    }
    const { nickname, region } = this.authService.getNickNameAndRegion();
    if (nickname && region) {
      this.nickName = nickname;
      this.region = region;
    }
  }

  redirectToLogin() {
    this.router.navigate(['/login']);
  }

  SignOut() {
    try {
      this.authService.clearToken();
      this.logoState = true;
      this.buttonState = false;
      this.router.navigate(['/']); // Redirect to home page or login page
      // this.snackbarService.success('Successfully signed out!', 'Close'); // Show success message
    } catch (error) {
      this.snackbarService.error('Error during sign-out. Please try again.', 'Close'); // Show error message
    }
  }

  navigateTo(page: string) {
    if (this.buttonState) {
      this.router.navigate([page]);
    } else {
      this.redirectToLogin();
    }
  }

  setActiveLink(link: string): void {
    this.activeLink = link;
  }

  
  goToAdvancedSearch() {
    this.router.navigate(['/fdac']);
  }

  
  onSearch() {
    if (!this.searchQuery || this.searchQuery.trim() === '') {
      // Show a snack bar message if the search field is empty
      // this.snackBar.open('Please enter a search query.', 'Close', { duration: 3000 });
      return;  // Stop the navigation if the search field is empty
    }
    // Always navigate to /search with the search query
    this.router.navigate(['/fdac'], { queryParams: { query: this.searchQuery } });
  }

  /**
   * Called when the input field gains focus.
   * Stops the animation cycle immediately.
   */
  onFocus(): void {
    // this.stopCycle();
  }
  
  /**
   * If the user leaves (blur) and the field is empty, restart the animation cycle.
   */
  onBlur(): void {
    if (!this.searchQuery) {
      // this.startCycle();
    }
  }

  onKeyDown(event: KeyboardEvent) {
    if (event.key === 'Enter') {
      this.onSearch();
    }
  }
}