import { Component, Input, OnInit, OnChanges, SimpleChanges } from '@angular/core';
import { Router } from '@angular/router';
import { GeneratedDataSearchService } from '../../services/generated-data-search.service';
import { SearchQueries, AssetsInteraction } from '../../interfaces/generated-data-search.interface';
import { forkJoin } from 'rxjs';
import { AssetWithOfferings } from '../../services/composite-search.service';
import { SearchResult } from '../../models/search-result.model';

type AssetWithState = AssetWithOfferings & { expanded?: boolean; hover?: boolean };

@Component({
  selector: 'app-search-results',
  templateUrl: './search-results.component.html',
  styleUrls: ['./search-results.component.scss']
})
export class SearchResultsComponent implements OnInit, OnChanges {

  set fdacResults(value: AssetWithState[]) {
    this._fdacResults = value;
    console.log('FDAC results in component:', this._fdacResults);
    this._fdacResults.forEach(i => {
      console.log(
        'dataElement.offerings:',
        i.asset?.dataElement?.offerings,
        'allOfferings:',
        i.allOfferings
      );
    });
  }  

  // Separate inputs for FDAC and Kaggle responses
  private _fdacResults: AssetWithState[] = [];
  private _kaggleResults: SearchResult[] = [];
  Math = Math;

  @Input()
  // set fdacResults(value: AssetWithState[]) {
  //   this._fdacResults = value;
  //   //this.debugFdacResults(value);  
  // }
  get fdacResults(): AssetWithState[] {
    return this._fdacResults;
  }

  @Input() loading = false;
  @Input() infoSearch: any;

  total: number = 0;
  searchId: string = '';
  currentPage: number = 1;
  pageSize: number = 18;

  constructor(
    private router: Router,
    private generatedDataSearchService: GeneratedDataSearchService
  ) { }

  ngOnInit(): void {
    this.currentPage = 1;
  }

  ngOnChanges(changes: SimpleChanges): void {
    // When search results change
    if (changes['fdacResults']) {
      this.total = this.fdacResults?.length || 0;
  
      // Reset to first page on every new search
      this.currentPage = 1;
    }
  
    // When infoSearch metadata changes (for logging/interactions)
    if (changes['infoSearch'] && this.infoSearch) {
      this.sendDataSearch();
    }
  }

  onMouseOver(event: Event): void {
    const target = event.target as HTMLElement;
    target.style.cursor = 'pointer';
  }

  onMouseOut(event: Event): void {
    const target = event.target as HTMLElement;
    target.style.cursor = 'default';
  }

  onTitleClick(item: any): void {
    if (!item?.asset?.id) {
      console.error('Error: item.asset.id is missing');
      return;
    }
    const url = this.router.serializeUrl(this.router.createUrlTree([`/asset/${item.asset.id}`]));
    window.open(url, '_blank');
    
    this.sendInteractionSearch([item], 'clicked');
  }

  private getFormattedTimestamp(): string {
    const now = new Date();
    const year = now.getFullYear();
    const month = String(now.getMonth() + 1).padStart(2, '0');
    const day = String(now.getDate()).padStart(2, '0');
    const hours = String(now.getHours()).padStart(2, '0');
    const minutes = String(now.getMinutes()).padStart(2, '0');
    const seconds = String(now.getSeconds()).padStart(2, '0');
    return `${year}-${month}-${day} ${hours}:${minutes}:${seconds}`;
  }

  private sendDataSearch(): void {
    const dataSearch: SearchQueries = {
      query: this.infoSearch.query,
      number_of_results: this.total,
      filters: this.infoSearch.filters,
      search_date: this.getFormattedTimestamp()
    };

    this.generatedDataSearchService.dataSearch(dataSearch).subscribe({
      next: (response) => {
        this.searchId = response?.search_id ? response.search_id : '';
        if (this.searchId) {
          this.sendInteractionSearch([...this.fdacResults], 'displayed');
        }
      },
      error: (error) => {
        const errorMessage = error?.message ? error.message : error;
        console.error('Error dataSearch : ', errorMessage);
      }
    });
  }

  private sendInteractionSearch(items: any[], interactionType: 'clicked' | 'displayed'): void {
    const interactions = items.map(item => {
      let interactionSearch: AssetsInteraction;

      if (item.source === 'KAGGLE' || item?.asset?.source === 'KAGGLE') {
        const kaggleItem = item.source === 'KAGGLE' ? item : item.asset;
        interactionSearch = {
          search_id: this.searchId,
          asset_id: kaggleItem.id || '',
          asset_name: kaggleItem.dataElement?.title || '',
          weight: item.asset?.weight ?? 0,
          interaction_type: interactionType,
          interaction_date: this.getFormattedTimestamp()
        };
      } else {
        interactionSearch = {
          search_id: this.searchId,
          asset_id: item.asset?.id || '',
          asset_name: item.asset?.dataElement?.name || '',
          weight: item.asset?.weight?.toFixed(2) || '0.00',
          interaction_type: interactionType,
          interaction_date: this.getFormattedTimestamp()
        };
      }

      return this.generatedDataSearchService.interactionSearch(interactionSearch);
    });

    forkJoin(interactions).subscribe({
      next: () => {},
      error: (error) => {
        const errorMessage = error?.message ? error.message : error;
        console.error('Error interactionSearch:', errorMessage);
      }
    });
  }

  showKaggle: boolean = false;
  toggleKaggle() {
    this.showKaggle = !this.showKaggle;
  }

  isCertified(item: any): boolean {
    const de = item?.asset?.dataElement ?? item?.dataElement;
    return !!de?.certification?.label; 
  }
  
  getCertificationLabel(item: any): string {
    const de = item?.asset?.dataElement ?? item?.dataElement;
    return de?.certification?.label || 'Certified';
  }

  private debugFdacResults(results: AssetWithState[]): void {
    console.log('FDAC:', results);
    results.forEach(item => {
      console.log('ID:', item.asset?.id);
      console.log('Name:', item.asset?.dataElement?.name);
      console.log('Model:', item.bestOffering?.model);
      console.log('Price:', item.bestOffering?.priceNum);
    });
  }

  get totalFdac(): number {
    return this.fdacResults?.length ?? 0;
  }

  get totalPages(): number {
    return Math.max(1, Math.ceil(this.totalFdac / this.pageSize));
  }

  get pagedFdacResults(): any[] {
    const start = (this.currentPage - 1) * this.pageSize;
    return (this.fdacResults ?? []).slice(start, start + this.pageSize);
  }

  get rangeStart(): number {
    if (!this.totalFdac) return 0;
    return (this.currentPage - 1) * this.pageSize + 1;
  }
  
  get rangeEnd(): number {
    return Math.min(this.currentPage * this.pageSize, this.totalFdac);
  }

  goToPage(p: number) {
    this.currentPage = Math.min(Math.max(1, p), this.totalPages);
  }
  
  nextPage() {
    if (this.currentPage < this.totalPages) this.currentPage++;
  }
  
  prevPage() {
    if (this.currentPage > 1) this.currentPage--;
  }

  getUniqueModels(offerings: any[]): string[] {
    if (!offerings || !Array.isArray(offerings)) return [];
    const models = offerings
      .map(o => o.model)
      .filter(m => typeof m === 'string' && m.trim().length > 0);
    return Array.from(new Set(models)); // remove duplicates
  }
}