import { Component, OnInit, AfterViewInit, inject } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { DOCUMENT, ViewportScroller } from '@angular/common';
import { AuthService } from 'src/app/auth/auth.service';
import { Observable, fromEvent, map } from 'rxjs';
@Component({
  selector: 'app-about-us-page',
  templateUrl: './about-us-page.component.html',
  styleUrls: ['./about-us-page.component.scss']
})
export class AboutUsPageComponent implements OnInit, AfterViewInit  {

  constructor(
    private route: ActivatedRoute,
    private viewportScroller: ViewportScroller,
    private authService: AuthService,
    private router: Router,
  ) { }

  ngOnInit(): void {
    // Only set selectedStepper if queryParam 'stepper' exists.
    this.route.queryParams.subscribe(params => {
      const step = +params['stepper'];
      if (step) {
        this.selectedStepper = step;
      }
    });
  }

  redirectToAboutUs(event: Event): void {
    event.preventDefault(); // Prevent any default behavior
    // Open /about-us with a query parameter 'stepper=1' in a new tab
    window.open('/about-us?stepper=1', '_blank');
  }

  redirectToAboutUs1(event: Event): void {
    event.preventDefault(); // Prevent any default behavior
    // Open /about-us with a query parameter 'stepper=2' in a new tab
    window.open('/about-us?stepper=2', '_blank');
  }

  get publishHref(): string {
    const tree = this.authService.isLoggedIn()
      ? this.router.createUrlTree(['/asset-publishment'])
      : this.router.createUrlTree(
          ['/login'],
          { queryParams: { returnUrl: 'asset-publishment' } }
        );
    // If you need an absolute URL, prefix with origin:
    // return window.location.origin + this.router.serializeUrl(tree);
    return this.router.serializeUrl(tree);
  }

  get profileHref(): string {
    const tree = this.authService.isLoggedIn()
      ? this.router.createUrlTree(['/profile-overview'])
      : this.router.createUrlTree(
          ['/login'],
          { queryParams: { returnUrl: 'profile-overview' } }
        );
    // you can prefix with origin if you want an absolute URL:
    return this.router.serializeUrl(tree);
  }

  get tradeHref(): string {
    const tree = this.authService.isLoggedIn()
      ? this.router.createUrlTree(['/trade-onboarding'])
      : this.router.createUrlTree(
          ['/login'],
          { queryParams: { returnUrl: 'trade-onboarding' } }
        );
    return this.router.serializeUrl(tree);
  }

  get catalogHref(): string {
    const tree = this.authService.isLoggedIn()
      ? this.router.createUrlTree(['/fdac'])
      : this.router.createUrlTree(
          ['/login'],
          { queryParams: { returnUrl: 'fdac' } }
        );
    // serializeUrl gives you "/fdac" or "/login?returnUrl=fdac"
    return this.router.serializeUrl(tree);
  }

  redirectToPage2(event: Event): void {
    event.preventDefault(); // Prevent default anchor behavior
  
    // Check if the user is logged in.
    if (!this.authService.isLoggedIn()) {
      // If not logged in, redirect to login with returnUrl set to '/trade-onboarding'
      this.router.navigate(['/login'], { queryParams: { returnUrl: 'trade-onboarding' } });
    } else {
      // If logged in, navigate directly to /trade-onboarding
      this.router.navigate(['/trade-onboarding']);
    }
  }

  // When null, no stepper is visible. Otherwise 1, 2 or 3.
  selectedStepper: number | null = null;

  selectStepper(step: number): void {
    // Toggle: if the same header is clicked, collapse; else, open new stepper.
    this.selectedStepper = this.selectedStepper === step ? null : step;
  }

  // Called when clicking the toggle button (stops propagation so the header's click isn’t triggered)
  toggleStepper(step: number, event: Event): void {
    event.stopPropagation();
    this.selectedStepper = this.selectedStepper === step ? null : step;
  }

  ngAfterViewInit(): void {
    this.route.fragment.subscribe(fragment => {
      if (fragment) {
        this.viewportScroller.scrollToAnchor(fragment);
      }
    });
  }
  
  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  onFinishStepper(): void {
    // Perform any final actions here, such as form submission or displaying a message.
    console.log('Stepper completed!');
  }

  showStepper: boolean = false;
  showStepper2: boolean = false;
  showStepper3: boolean = false;

  toggleStepper2(): void {
    this.showStepper2 = !this.showStepper2;
  }

  toggleStepper3(): void {
    this.showStepper3 = !this.showStepper3;
  }
}