import { AfterViewInit, Component, OnDestroy, OnInit, ViewChild, inject } from '@angular/core';
import { NgForm } from '@angular/forms';
import { MatSnackBar, MatSnackBarRef, SimpleSnackBar } from '@angular/material/snack-bar';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { Router, ActivatedRoute } from '@angular/router';
import { GovernanceApiService } from '../../services/api/governance.api.service';
import { Observable, finalize, fromEvent, map, take, lastValueFrom, firstValueFrom } from 'rxjs';
import { environment } from '../../../environments/environment.prod';
import { SnackbarService } from '../../services/snackbar/snackbar.service';
import { AuthService } from '../../auth/auth.service';
import { MarkdownPreviewService, MarkdownPreviewResult } from '../../services/markdown-preview/markdown-preview.service';
import { MatDialog } from '@angular/material/dialog';
import { MarkdownPreviewModalComponent } from '../../components/markdown-preview-modal/markdown-preview-modal.component';
import { EditAssetConfirmationDialogComponent } from '../../components/edit-asset-confirmation-dialog/edit-asset-confirmation-dialog.component';
import { DOCUMENT, ViewportScroller } from '@angular/common';

@Component({
  selector: 'app-asset-publishment',
  templateUrl: './asset-publishment.component.html',
  styleUrls: ['./asset-publishment.component.scss'],
})
export class AssetPublishmentComponent implements OnInit, OnDestroy, AfterViewInit {
  useMockup: boolean = false;

  private accounts: any = null;
  private defaultDuration: number = 3000;
  protected txHash: any;

  constructor(
    private snackBar: MatSnackBar,
    private metamaskService: MetamaskChainService,
    private provenanceApiService: ProvenanceApiService,
    private governanceApiService: GovernanceApiService,
    private clipboardManager: Clipboard,
    private router: Router,
    private route: ActivatedRoute,
    private snackbarService: SnackbarService,
    private authService: AuthService,
    private markdownPreviewService: MarkdownPreviewService,
    private dialog: MatDialog
  ) {
    this.connectToMetamask();
  }

  @ViewChild('assetForm') assetForm!: NgForm;
  asset = {
    dcterms_title: '',
    dcterms_description_short: '',
    dcterms_type: '',
    dcat_endpointURL: '',
    dcat_landingPage: '',
    // dcterms_conformsTo: '', // Commented out - backend doesn't use this value
    dcterms_description_long: '',
    fame_logo: '',
  };
  types: string[] = [
    'DATA',
    'MODEL',
    'DOCUMENT',
    'APPLICATION',
    'SERVICE',
    'WEBAPP',
    'CLIENT',
    'LIBRARY',
  ];
  isMetamaskConnected: boolean = false;
  isSubmitted: any = false;
  assetId: any = '';
  copyToClipboard: any;
  subscribeReq: any;
  subscribePt: any;

  // Edit mode properties
  isEditMode: boolean = false;
  currentAid: string | null = null;
  originalAssetData: any = null;
  isLoadingAssetData: boolean = false;
  showNewAssetBanner: boolean = false;
  isMetaMaskPending: boolean = false;

  ngOnInit() {
    this.checkLoginStatus();
    localStorage.removeItem('offering');
    this.initializeEditMode();

    // Listen for route parameter changes to handle navigation within the same component
    this.route.params.subscribe(params => {
      console.log('Route params changed:', params);
      this.initializeEditMode();
    });
  }

  ngOnDestroy(): void {
    if (this.subscribePt) this.subscribePt.unsubscribe();
    if (this.subscribeReq) this.subscribeReq.unsubscribe();
  }

  ngAfterViewInit(): void {
    const script = document.createElement('script');
    script.src = 'https://website-widgets.pages.dev/dist/sienna.min.js';
    script.async = true;  // or false, depending on your needs
  
    script.onload = () => {
      console.log('Sienna script loaded!');
      // Optionally, call any global functions provided by the script
      // For example: window['Sienna']?.init(...);
    };
  
    // Append the script element to the document head (or body)
    document.head.appendChild(script);
  }

  checkLoginStatus(): void {
    if (!this.authService.getToken() || this.authService.isTokenExpired()) {
      // Optionally show a message indicating that login is required
      this.snackbarService.error('Please log in to access this page.', 'Close');
      this.router.navigate(['/login']);
    }
  }

  /**
   * Initialize edit mode if aid parameter is present in route
   */
  private async initializeEditMode(): Promise<void> {
    this.currentAid = this.route.snapshot.paramMap.get('aid');

    if (this.currentAid) {
      // Check if this is certification page vs edit mode
      const isNewAsset = this.route.snapshot.queryParams['newAsset'] === 'true';

      if (isNewAsset) {
        // This is the certification page - determine edit mode from the action parameter
        const action = this.route.snapshot.queryParams['action'];
        this.isEditMode = action === 'republish';
      } else {
        // This is regular edit mode (loading existing asset for editing)
        this.isEditMode = true;
      }
      this.isLoadingAssetData = true;

      if (isNewAsset) {
        // This is the certification page - load asset data and handle MetaMask
        console.log('Certification page detected:', {
          isNewAsset,
          currentAid: this.currentAid,
          queryParams: this.route.snapshot.queryParams,
          isEditMode: this.isEditMode
        });

        try {
          // First load the asset data to populate the form
          const assetData = await lastValueFrom(this.provenanceApiService.getAssetDetails(this.currentAid));
          this.loadAssetDataForEdit(assetData);

          // Set loading to false so the page content is visible
          this.isLoadingAssetData = false;

          console.log('About to call handleNewAssetCertification');
          // Handle MetaMask certification
          await this.handleNewAssetCertification();

        } catch (error: any) {
          console.error('Error handling asset certification:', error);
          this.isLoadingAssetData = false;

          if (error.status === 404) {
            this.snackbarService.error('Asset not found or you do not have access to view it.', 'Close');
          } else {
            this.snackbarService.error('Error loading asset data. Please try again.', 'Close');
          }

          this.router.navigate(['/']);
        }
      } else if (this.isEditMode) {
        // This is regular edit mode - load asset data for editing
        try {
          // First check asset existence and visibility
          const assetData = await lastValueFrom(this.provenanceApiService.getAssetDetails(this.currentAid));

          // Then check if user can edit this asset
          const editPermission = await lastValueFrom(this.provenanceApiService.checkAssetForResubmission(this.currentAid));

          if (!editPermission.canEdit) {
            this.snackbarService.error(editPermission.reason || 'You do not have permission to edit this asset.', 'Close');
            this.router.navigate(['/']);
            return;
          }

          // Load asset data for editing
          this.loadAssetDataForEdit(assetData);

          // Set loading to false
          this.isLoadingAssetData = false;

        } catch (error: any) {
          console.error('Error loading asset data for edit:', error);
          this.isLoadingAssetData = false;

          if (error.status === 404) {
            this.snackbarService.error('Asset not found or you do not have access to view it.', 'Close');
          } else if (error.status === 401 || error.status === 403) {
            this.snackbarService.error('You do not have permission to edit this asset.', 'Close');
          } else {
            this.snackbarService.error('Error loading asset data. Please try again.', 'Close');
          }

          this.router.navigate(['/']);
        }
      }
    }
  }

  /**
   * Load and map asset data from API response to form fields
   */
  private loadAssetDataForEdit(apiResponse: any): void {
    // Store original data for comparison
    this.originalAssetData = { ...this.asset };

    // Extract endpoint URL from "Platform@https://..." format
    const endpointUrl = apiResponse.dcat_endpointURL?.split('@')[1] || '';

    // Map API response to form fields
    this.asset = {
      dcterms_title: apiResponse.dcterms_title || '',
      dcterms_description_short: apiResponse.dcterms_description_short || '',
      dcterms_type: this.reverseMapAssetType(apiResponse.dcterms_type) || '',
      dcat_endpointURL: endpointUrl,
      dcat_landingPage: apiResponse.dcat_landingPage || '',
      // dcterms_conformsTo: this.formatStandardsArray(apiResponse.dcterms_conformsTo) || '', // Commented out - backend doesn't use this value
      dcterms_description_long: this.convertHtmlToMarkdown(apiResponse.dcterms_description_long) || '',
      fame_logo: apiResponse.fame_logo || '',
    };

    // Store the mapped data as original for comparison
    this.originalAssetData = { ...this.asset };
  }

  /**
   * Reverse map asset type from catalogue type to form type
   * Note: V2 API returns lowercase types
   */
  private reverseMapAssetType(catalogueType: string): string {
    const typeMapping: { [key: string]: string } = {
      'dataset': 'DATA',
      'model': 'MODEL',
      'documentation': 'DOCUMENT',
      'application': 'APPLICATION',
      'web - api': 'SERVICE',
      'web-based tool': 'WEBAPP',
      'client': 'CLIENT',
      'library': 'LIBRARY'
    };

    return typeMapping[catalogueType.toLowerCase()] || catalogueType;
  }

  /**
   * Convert standards array to comma-separated string
   */
  private formatStandardsArray(standards: any[]): string {
    if (!standards || !Array.isArray(standards)) {
      return '';
    }
    return standards.join(', ');
  }

  /**
   * Convert HTML content back to Markdown for editing
   * IMPORTANT: This is a lossy conversion that can cause issues with republishing.
   * We should avoid this when possible by storing original markdown or detecting
   * when content might not convert properly.
   */
  private convertHtmlToMarkdown(htmlContent: string): string {
    if (!htmlContent) {
      return '';
    }

    return this.markdownPreviewService.htmlToMarkdown(htmlContent);
  }


  /**
   * Check if current form data differs from original data
   */
  private hasFormChanges(): boolean {
    if (!this.originalAssetData) {
      return false;
    }

    return JSON.stringify(this.asset) !== JSON.stringify(this.originalAssetData);
  }

  /**
   * Handle MetaMask certification for newly created assets
   */
  private async handleNewAssetCertification(): Promise<void> {
    // Check if this is a newly created asset (from query params)
    const isNewAsset = this.route.snapshot.queryParams['newAsset'] === 'true';

    console.log('handleNewAssetCertification called:', {
      isNewAsset,
      hasAccounts: this.accounts && this.accounts.length > 0,
      accounts: this.accounts,
      queryParams: this.route.snapshot.queryParams
    });

    if (isNewAsset && this.accounts && this.accounts.length > 0) {
      // Show the informative banner
      this.showNewAssetBanner = true;
      console.log('Banner should be shown, showNewAssetBanner set to:', this.showNewAssetBanner);

      try {
        // Retrieve the stored API response data for MetaMask transaction
        const storedDataStr = localStorage.getItem('newAssetData');
        if (!storedDataStr) {
          console.error('No stored asset data found for MetaMask transaction');
          return;
        }

        const transactionData = JSON.parse(storedDataStr);
        console.log('Triggering MetaMask for newly created asset:', this.currentAid);

        // Set MetaMask pending state
        this.isMetaMaskPending = true;

        // Trigger MetaMask transaction with the original API response data
        const transactionHash = await this.metamaskService.executeAssetTransaction(transactionData, this.accounts[0]);

        console.log('Asset certified via MetaMask', transactionHash);
        this.txHash = transactionHash;

        this.snackbarService.success(
          `Asset certified! Transaction ID: ${transactionHash}`,
          'Close',
          3000
        );

        // Clean up stored data and navigate to asset details page
        localStorage.removeItem('newAssetData');
        this.router.navigate(['/asset', this.currentAid]);

      } catch (error: any) {
        console.error('MetaMask certification error:', error);

        // Show user-friendly message - asset is still published, just not certified
        this.snackbarService.info(
          'Asset published successfully. Certification was cancelled or failed. You can try again later.',
          'Close',
          5000
        );

        // Clean up stored data and navigate to asset details page
        localStorage.removeItem('newAssetData');
        this.router.navigate(['/asset', this.currentAid]);
      }
    }
  }

  /**
   * Dismiss the new asset banner
   */
  dismissBanner(): void {
    this.showNewAssetBanner = false;
  }

  /**
   * Abort the certification process and navigate to catalogue view
   */
  abortCertification(): void {
    // Clean up stored data
    localStorage.removeItem('newAssetData');

    // Navigate to asset details page
    this.router.navigate(['/asset', this.currentAid]);
  }

  onDiscard() {
    if (this.isEditMode && this.originalAssetData) {
      // Reset to original loaded data
      this.asset = { ...this.originalAssetData };
      // Force form validation update
      setTimeout(() => {
        if (this.assetForm) {
          this.assetForm.form.markAsUntouched();
          this.assetForm.form.markAsPristine();
        }
      }, 0);
    } else {
      // Clear form for new asset
      this.assetForm.resetForm();
      this.router.navigate(['/']).then(() => window.scrollTo(0, 0));
    }
  }

  backendMessages = {
    assetTitle: 'Provide a concise title of the catalogue entry. Use Latin letters, numbers, punctuation, and spaces only. Maximum length: 80 characters (Required)',
    assetType: 'Select the asset type (Required)',
    shortDescription: 'Provide a brief description of the catalogue entry. Use Latin letters, numbers, punctuation, spaces, and newlines only. Maximum length: 500 characters (Required)',
    fullDescription: 'Provide a full description of the catalogue entry. Basic Markdown formatting is supported: **bold**, *italic*, # headings, and lists. Maximum length: 5000 characters (Optional)',
    assetLogo: 'Provide a valid URL for the asset logo. Must start with http:// or https:// (Optional)',
    assetWebSite: 'Provide a valid URL for the asset website. Must start with http:// or https:// (Optional)',
    contentDelivery: 'Provide the asset delivery point URL for buyers to retrieve the asset content. Must start with http:// or https:// (Required)'
  };

  ONE_LINER_MAXL = 80;
  TXT_SHORT_MAXL = 500;
  TXT_LONG_MAXL = 5000;
  urlListPattern = '^\\s*(https?:\\/\\/[^\\s,]+)(?:\\s*,\\s*https?:\\/\\/[^\\s,]+)*\\s*$';


  isLoading = false;


  async onSubmit() {
    // 1) Check if form is valid
    if (!this.assetForm.valid) {
      this.snackbarService.info('Please fill all the required fields', 'Close', 3000);
      return;
    }

    // 2) In edit mode, check for changes and show confirmation dialog
    if (this.isEditMode) {
      if (!this.hasFormChanges()) {
        this.snackbarService.info('You made no changes to the current data!', 'Close', 3000);
        return;
      }

      // Show confirmation dialog
      const dialogRef = this.dialog.open(EditAssetConfirmationDialogComponent, {
        width: '600px',
        data: {
          assetTitle: this.asset.dcterms_title
        }
      });

      const confirmed = await firstValueFrom(dialogRef.afterClosed());
      if (!confirmed) {
        return; // User cancelled
      }
    }

    console.log('Form Submitted', this.asset);
    this.isLoading = true;

    // 3) Submit asset to the backend or provenance API
    const normalizedAsset = this.getNormalizedAsset(); // needed to comply with strict validity checks on the BE

    // Use different API method based on mode
    const apiCall = this.isEditMode
      ? this.provenanceApiService.resubmitAsset(this.currentAid!, normalizedAsset)
      : this.provenanceApiService.submitAsset(normalizedAsset);

    this.subscribePt = apiCall.pipe(finalize(() => (this.isLoading = false))).subscribe({
      next: async (data: any) => {
        if (data) {
          console.log('Asset submission result:', data);
          this.assetId = data.aid;

          if (this.isEditMode) {
            // For edit mode, use the same UX flow as new assets
            this.snackbarService.success('Asset updated successfully', 'Close', 3000);

            // Store the complete API response for MetaMask transaction on re-publishing page
            localStorage.setItem('newAssetData', JSON.stringify(data));

            // Navigate to certification page with action=republish using the NEW asset ID
            this.router.navigate(['/asset-publishment', data.aid], {
              queryParams: { newAsset: 'true', action: 'republish' }
            });
          } else {
            // For new assets: redirect immediately to certification page
            // MetaMask interaction will happen on the certification page
            this.snackbarService.success('Asset published successfully', 'Close', 3000);

            // Store the complete API response for MetaMask transaction on certification page
            localStorage.setItem('newAssetData', JSON.stringify(data));

            // Navigate to certification page with action=publish using the NEW asset ID
            this.router.navigate(['/asset-publishment', data.aid], {
              queryParams: { newAsset: 'true', action: 'publish' }
            });
          }
        } else {
          // If 'data' is somehow empty
          this.snackbarService.error(
            'No data returned from server. Asset creation failed.',
            'Close',
            5000
          );
        }
      },
      error: (err: any) => {
        // 2b) ERROR from the API service
        console.error('submitAsset error:', err);
        this.handleValidationError(err);
      },
    });
  }

  private handleValidationError(error: any): void {
    let errorMessage = 'An error occurred while processing your request.';
    
    if (error.status === 400 && error.error) {
      // Extract class-validator messages
      if (error.error.message && Array.isArray(error.error.message)) {
        // Join validation messages with line breaks
        errorMessage = error.error.message.join('\n');
      } else if (typeof error.error.message === 'string') {
        errorMessage = error.error.message;
      } else if (error.error.details) {
        errorMessage = error.error.details;
      }
    } else if (error.message) {
      errorMessage = error.message;
    }
    
    this.snackbarService.error(errorMessage, 'Close', 5000);
    this.isSubmitted = false; // Allow user to retry
    this.isLoading = false; // Reset loading state on error
  }

  public async connectToMetamask() {
    // @ts-ignore
    if (typeof window.ethereum === 'undefined' || !window.ethereum) {
      this.snackBar.open(
        'MetaMask plugin is not installed!',
        'Please install MetaMask',
        {
          duration: this.defaultDuration,
        }
      );
      this.isMetamaskConnected = false;
    } else {
      await this.getMetamaskAccounts();
      if (this.accounts == null) return;
      this.isMetamaskConnected = true;
      this.snackBar.open('MetaMask is connected!', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private async getMetamaskAccounts() {
    try {
      // @ts-ignore
      const accounts = await window.ethereum.request({
        method: 'eth_requestAccounts',
      });
      console.log('Accounts: ' + accounts);
      this.accounts = accounts;
    } catch (err) {
      // @ts-ignore
      if (err.code === 4001) {
        // EIP-1193 userRejectedRequest error
        // If this happens, the user rejected the connection request.
        // this.showToast(this.toastService, 'Please connect to MetaMask.', 'warning');
        this.snackBar.open('Please connect to MetaMask.', 'Warn', {
          duration: this.defaultDuration,
        });
      } else {
        this.snackBar.open(err + '', 'Error', {
          duration: this.defaultDuration,
        });
      }
    }
  }

  copyAssetUri(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.assetId);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    this.subscribeReq = this.governanceApiService
      .checkRequestStatus(this.assetId)
      .pipe(take(1))
      .subscribe(
        (response: any) => {
          if (!response) return;
          console.log('Response', response);
          this.snackBar.open('Submitted Request:', response.message, {
            duration: 6000,
          });
        },
        (error) => {
          console.log('Error', error);
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        }
      );
  }

  goToFdac($event: any) {
    $event.preventDefault();
    this.snackBar.open('Redirecting to FDAC...', 'Info', {
      duration: this.defaultDuration,
    });
    window.open(`${environment.marketplaceUrl}/asset/${this.assetId}`, '_blank');
  }

  goToCatalogue($event: any) {
    $event.preventDefault();
    this.router.navigate(['/asset/' + this.assetId]);
  }

  openEditPolicy(assetId: string) {
    const url = `${environment.apmUIURL}/asset/${assetId}`;
    window.open(url, '_blank');
  }

  private getNormalizedAsset() {
    let normalizedAsset = JSON.parse(JSON.stringify(this.asset));
    for (const key in normalizedAsset) {
      if (normalizedAsset.hasOwnProperty(key)) {
        const value = normalizedAsset[key];
        // Check if the value is null/undefined or if it's a string that is empty or only whitespace.
        if (value == null || (typeof value === 'string' && value.trim() === ''))
          delete normalizedAsset[key];
      }
    }
    return normalizedAsset;
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  redirectToAboutUs(event: Event): void {
    event.preventDefault(); // Prevent any default behavior
    // Open /about-us with a query parameter 'stepper=1' in a new tab
    window.open('/about-us?stepper=3', '_blank');
  }

  /**
   * Handle Markdown preview for the long description field
   */
  previewMarkdown(): void {
    const result: MarkdownPreviewResult = this.markdownPreviewService.previewMarkdown(
      this.asset.dcterms_description_long,
      this.TXT_LONG_MAXL
    );

    if (!result.success) {
      // Show toast with appropriate message
      this.snackbarService.info(result.message || 'Preview not available', 'Close', 3000);
      return;
    }

    // Show modal with formatted content
    this.dialog.open(MarkdownPreviewModalComponent, {
      width: '80%',
      maxWidth: '800px',
      maxHeight: '80vh',
      data: {
        html: result.html,
        title: 'Full Description Preview'
      }
    });
  }
}