import {
  Component,
  HostListener,
  OnInit,
  OnDestroy,
  ViewChild,
  ElementRef
} from '@angular/core';
import { ActivatedRoute, ParamMap, Router } from '@angular/router';
import { switchMap, map, catchError, distinctUntilChanged } from 'rxjs/operators';
import { from, of } from 'rxjs';
import {
  CompositeSearchService,
  AssetWithOfferings,
} from 'src/app/services/composite-search.service';
import { AuthService } from 'src/app/auth/auth.service';
//import { AdvancedSearchService } from 'src/app/services/advanced-search.service';
import { Subject, Subscription } from 'rxjs';
import { debounceTime, finalize } from 'rxjs/operators';
import { OpenAPIService } from 'src/app/services/open-api.service';
import { PolicyIntegrationService } from 'src/app/services/policy-integration.service'

@Component({
  selector: 'app-fdac-search',
  templateUrl: './fdac-search.component.html',
  styleUrls: ['./fdac-search.component.scss'],
})
export class FdacSearchComponent implements OnInit, OnDestroy {
    /* filters */
  query = '';

  private autoSearch$ = new Subject<void>();
  private autoSub?: Subscription;

  businessModels = [
    { name: 'ALL_ASSETS', label: 'All Assets' },
    { name: 'ALL_BM', label: 'Paid Plans (SUB/PAYG/PAYU)' },
    { name: 'SUB', label: 'Subscription' },
    { name: 'PAYG', label: 'Pay-As-You-Go' },
    { name: 'PAYU', label: 'Pay-As-You-Use' },
    { name: 'FREE', label: 'Free of charge' },
  ];
  selectedBusinessModel: { name: string; label: string } | null = null;
  selectedBusinessModels: string[] = [];


  publisher = '';
  assetTypes = [
    'All', 'Dataset', 'Model', 'Documentation', 'Application',
    'Web - API', 'Web-based tool', 'Client', 'Library',
  ];
  selectedAssetTypes: string[] = [];

  freeAssetsOnly = false;
  showCertifiedOnly = false;
  isMobile = window.innerWidth <= 768;
  loading = false;

  /* auth state */
  isAuthenticated = false;

  /* results */
  fdacResponse: AssetWithOfferings[] = [];

  sortOrder: 'asc' | 'desc' = 'asc';
  private _prevAssetTypes: string[] = [...this.selectedAssetTypes];

  subscriptionPriceRange = { min: 0, max: 1000 };  // default range
  enablePriceFilter = false;
  hasSearched = false;


  filtersOpen: boolean = false;
  dropdownOpen = false;
  dropdownUserOpen = false;
  @ViewChild('filters') filters!: ElementRef;

  constructor(
    private composite: CompositeSearchService,
    private route: ActivatedRoute,
    private router: Router,
    private auth: AuthService,
    private openAPI: OpenAPIService,
    private policy: PolicyIntegrationService
  ) {}

  ngOnInit(): void {
    this.isAuthenticated = this.auth.isLoggedIn();

    this.route.queryParamMap.pipe(
      map((p: ParamMap) => (p.get('query') ?? '').trim()),
      distinctUntilChanged()
    ).subscribe(q => {
      this.query = q;
      this.onFilterChange?.();
      this.search();
    })

    this.autoSub = this.autoSearch$
      .pipe(debounceTime(300))
      .subscribe(() => this.search());
  }

  ngOnDestroy(): void {
    this.autoSub?.unsubscribe();
    this.router.navigate(
      [],
      { queryParams: { query: null }, queryParamsHandling: 'merge' }
    );
  }

  onFilterChange(): void {
    this.autoSearch$.next();
  }

  certifiedOnlyChange() {
    this.search();
  }

  setSortOrder(order: 'asc' | 'desc') {
    this.sortOrder = order;
    this.applySorting();
  }

  search(): void {
    this.loading = true;

    const filters: { type: string; values: string[] }[] = [];

    if (this.selectedAssetTypes.length && !this.selectedAssetTypes.includes('All')) {
      filters.push({ type: 'tag', values: this.selectedAssetTypes });
    }

    if (this.publisher.trim()) {
      filters.push({ type: 'developer', values: [this.publisher.trim()] });
    }

    if (this.selectedBusinessModel) {
      const modelName = this.selectedBusinessModel.name;
    
      
      if (modelName === 'ALL_BM') {
        // Add all different business models explicitly
        this.selectedBusinessModels = this.businessModels.map(bm => bm.name);
        
      }
      else {
        // Regular specific filter
        this.selectedBusinessModels = [modelName];
        if(this.selectedBusinessModel.name === 'ALL_ASSETS'){
          this.selectedBusinessModels = [];
        }
      }
    }

    /* keep the URL in sync */
    this.router.navigate(
      [],
      { queryParams: { query: this.query }, queryParamsHandling: 'merge' }
    );

    this.composite
      .searchAssetsWithPrice(
        this.query ? this.query : '*',
        filters,
        this.selectedBusinessModels,
      ).pipe(
        finalize(() => this.loading = false))
      .subscribe({
        next: (assets) => {
          
          const bm = this.selectedBusinessModel?.name?.trim().toLowerCase();

          if (bm && bm !== 'all_assets') {
            assets = assets.filter(a => (a.allOfferings?.length ?? 0) > 0);
          }

          if (this.showCertifiedOnly) {
            assets = assets.filter(a => !!a.asset?.dataElement?.certification?.label);
          }

          if (this.selectedBusinessModel?.name === 'SUB' && this.enablePriceFilter) {
            const { min, max } = this.subscriptionPriceRange;
            assets = assets.filter(a => {
              const price = a.bestOffering?.priceNum ?? null;
              return price !== null && price >= min && price <= max;
            });
          }

          this.fdacResponse = assets;
          this.loading = false;
          this.applySorting();
          this.hasSearched = true;
        },
        error: err => {
          console.error(err);
          this.loading = false;
        },
      });
  }
  
  clearSearchInput(): void {
    this.query = '';
    this.hasSearched = false;
    this.fdacResponse = [];
    this.search();
  }

  clearFilters(): void {
    this.query = '';

    this.selectedBusinessModel = null;
    this.selectedAssetTypes = [];
    this.publisher = '';
    this.freeAssetsOnly = false;
    this.fdacResponse = [];
    this.subscriptionPriceRange = { min: 0, max: 1000 };
    this.enablePriceFilter = false;
    this.showCertifiedOnly = false;
    this.router.navigate(
      [],
      { queryParams: { query: null }, queryParamsHandling: 'merge' }
    );
    this.search();
  }

  @HostListener('window:resize')
  onResize() {
    this.isMobile = window.innerWidth <= 768;
  }

  applySorting() {
    if (!this.fdacResponse) return;

    this.fdacResponse = [...this.fdacResponse].sort((a, b) => {
      const nameA = a.asset?.dataElement?.name?.toLowerCase() || '';
      const nameB = b.asset?.dataElement?.name?.toLowerCase() || '';

      if (this.sortOrder === 'asc') return nameA.localeCompare(nameB);
      if (this.sortOrder === 'desc') return nameB.localeCompare(nameA);
      return 0;
    });
  }

  onAssetTypeChange(newSelected: string[]) {
    const hadAll = this._prevAssetTypes.includes('All');
    const hasAll = newSelected.includes('All');

    if (hasAll && !hadAll) {
      this.selectedAssetTypes = this.assetTypes;
    }
    else if (hasAll && hadAll && newSelected.length > 1) {
      this.selectedAssetTypes = newSelected.filter(t => t !== 'All');
    }
    else if (!hasAll && hadAll) {
      this.selectedAssetTypes = [];
    }
    else {
      this.selectedAssetTypes = newSelected.filter(Boolean);
    }

    this._prevAssetTypes = [...this.selectedAssetTypes];
    this.onFilterChange();
  }

  toggleFilters() {
    this.filtersOpen = !this.filtersOpen;
    if (this.filtersOpen) {
      document.body.style.overflow = 'hidden'; // Prevent background scrolling
    } else {
      document.body.style.overflow = 'auto';
    }
  }

  @HostListener('document:click', ['$event'])
  handleDocumentClick(event: Event) {
    const target = event.target as HTMLElement;
    const clickedInsideSort = target.closest('.dropdown.sort');
    const clickedInsideUser = target.closest('.dropdown.user');
    const clickedInsideFilters = this.filters.nativeElement.contains(target);
    const clickedFilterToggle = target.closest('.filter-toggle-btn');

    if (!clickedInsideSort) {
      this.dropdownOpen = false;
    }

    if (!clickedInsideUser) {
      this.dropdownUserOpen = false;
    }

    // Close filters if clicked outside and not on the toggle button
    if (this.filtersOpen && !clickedInsideFilters && !clickedFilterToggle) {
      this.closeFilters();
    }
  }

  closeFilters() {
    this.filtersOpen = false;
    document.body.style.overflow = 'auto';
  }
}