import {
  AfterViewInit,
  Component,
  ElementRef,
  HostListener,
  OnInit,
  ViewChild,
} from '@angular/core';
import { ActivatedRoute } from '@angular/router';
import { ComponentList } from 'src/app/components/component-list/component-list.component';
import { TagQueryEntry } from 'src/app/components/element-list/element-list.component';

interface Item {
  title: string;
}

@Component({
  selector: 'app-fdac',
  templateUrl: './fdac.component.html',
  styleUrls: ['./fdac.component.scss'],
})
export class FdacComponent implements OnInit, AfterViewInit {
  items: Item[] = Array.from({ length: 100 }, (_, i) => ({
    title: `AI based detection of fraudulent immediate loan requests ${i + 1}`,
  }));
  paginatedItems: Item[] = [];

  // Note: we no longer need both selectedSort & sortOrder; we use sortOrder ('asc'|'desc') to sort by “name”
  searchQuery = ''; // Search query
  initialSearchQuery = '';

  // 4) “Types” filter must match exactly the labels from “Publish with us”:
  //    Replace this array with the exact strings from your home‐page.
  types = [
    'All',
    'Dataset',
    'Model',
    'Documentation',
    'Application',
    'Web - API',
    'Web-based tool',
    'Client',
    'Library',
  ];
  typeHM: { [k: string]: string } = {
    [this.types[1]]: 'Dataset',
    [this.types[2]]: 'Model',
    [this.types[3]]: 'Documentation',
    [this.types[4]]: 'Application',
    [this.types[5]]: 'Web - API',
    [this.types[6]]: 'Web-based tool',
    [this.types[7]]: 'Client',
    [this.types[8]]: 'Library'
  }


  // Keep track of which chips are selected. Default = ['All']
  selectedTypes = ['All'];

  // Date range
  dateFrom: Date | null = null;
  dateTo: Date | null = null;

  // Sorting: ascending by name or descending by name
  sortOrder: 'asc' | 'desc' = 'asc';

  // Pagination (not touched by filters; you can adjust as needed)
  selectedPageSize = '18';
  currentPage = 0;

  componentList: ComponentList | null = null;
  filtersOpen: boolean = false; // for mobile toggle

  @ViewChild('components')
  set setComponentList(el: ComponentList) {
    this.componentList = el;
  }

  @ViewChild('filters') filters!: ElementRef;
  @ViewChild('searchInput') searchInput!: ElementRef;

  // Variable used to compare the current and the new type selection
  previousSelectedTypes: string[] = this.selectedTypes

  constructor(private route: ActivatedRoute) {
    this.onTypeChange = this.onTypeChange.bind(this)
  }

  ngOnInit(): void {
    this.route.queryParams.subscribe((params) => {
      if (params['query']) {
        this.searchQuery = params['query'];
        // this.initialSearchQuery = params['query'];
      }
    });
    this.updatePaginatedItems();
  }

  ngAfterViewInit() {
    if (this.searchQuery) {
      setTimeout(() => {
        // Simulate Enter key press to trigger initial search
        const event = new KeyboardEvent('keydown', { key: 'Enter' });
        this.searchInput.nativeElement.dispatchEvent(event);
      }, 7000);
    }
  }

  /** Triggered on Enter key in search field */
  onChange(ev: any) {
    this.triggerSearch(ev.target.value);
  }

  triggerSearch(query: string) {
    this.componentList?.search(query.trim());
  }

  toggleFilters() {
    this.filtersOpen = !this.filtersOpen;
  }

  @HostListener('document:click', ['$event'])
  handleDocumentClick(event: Event) {
    const target = event.target as HTMLElement;
    const clickedInsideFilters = this.filters.nativeElement.contains(target);
    const clickedFilterToggle = target.closest('.filter-toggle-btn');

    // Close filters if clicked outside
    if (this.filtersOpen && !clickedInsideFilters && !clickedFilterToggle) {
      this.filtersOpen = false;
    }
  }

  updatePaginatedItems() {
    const pageSize = parseInt(this.selectedPageSize, 10);
    const startIndex = this.currentPage * pageSize;
    const endIndex = startIndex + pageSize;
    this.paginatedItems = this.items.slice(startIndex, endIndex);
  }

  onPageSizeChange(newSize: string) {
    this.selectedPageSize = newSize;
    this.currentPage = 0;
    this.updatePaginatedItems();
  }

  onPageChange(event: any) {
    this.currentPage = event.pageIndex;
    this.updatePaginatedItems();
  }

  /**
   * 5) Fix “All” vs. individual‐type selection logic.
   *    - If the user clicks “All”, deselect everything else and keep only “All”.
   *    - If they pick any other type, remove “All” (so “All” cannot coexist with a specific type).
   */
  onTypeChange(selectedTypes: string[]) {
    const allCombinations = [
      ['Datasets', 'Algorithms', 'Software'],
      ['Datasets', 'Algorithms'],
      ['Documentations', 'Tutorials'],
      ['Datasets', 'Algorithms', 'Software', 'Documentations'],
      ['Datasets', 'Algorithms', 'Software', 'Tutorials'],
      ['Datasets', 'Algorithms', 'Documentations'],
      ['Datasets', 'Algorithms', 'Documentations', 'Tutorials'],
      ['Datasets', 'Algorithms', 'Tutorials'],
      ['Datasets', 'Software'],
      ['Datasets', 'Software', 'Documentations'],
      ['Datasets', 'Software', 'Documentations', 'Tutorials'],
      ['Datasets', 'Software', 'Tutorials'],
      ['Datasets', 'Documentations'],
      ['Datasets', 'Documentations', 'Tutorials'],
      ['Datasets', 'Tutorials'],
      ['Algorithms', 'Software'],
      ['Algorithms', 'Software', 'Documentations'],
      ['Algorithms', 'Software', 'Documentations', 'Tutorials'],
      ['Algorithms', 'Software', 'Tutorials'],
      ['Algorithms', 'Documentations'],
      ['Algorithms', 'Documentations', 'Tutorials'],
      ['Algorithms', 'Tutorials'],
      ['Software', 'Documentations'],
      ['Software', 'Documentations', 'Tutorials'],
      ['Software', 'Tutorials'],
      ['Documentations', 'Tutorials']
    ];

    const setSelectedTypes = (types: string[]) => {
      this.selectedTypes = types;

      this.previousSelectedTypes = types

      let tagQuery: TagQueryEntry[] | null = null
      if (this.selectedTypes.length > 1 || this.selectedTypes[0] !== "All") {
        tagQuery = this.selectedTypes.map(el => ({
          name: this.typeHM[el],
          type: "componentType"
        }))
      }
      console.log({ tagQuery })
      this.componentList?.filterByTags(tagQuery)

    };

    const shouldSelectAll = (types: string[]): boolean => {
      return allCombinations.some(combination =>
        combination.every(type => types.includes(type)) &&
        combination.length === types.length
      );
    };

    if (
      !this.previousSelectedTypes.includes("All") && this.selectedTypes.includes("All") ||
      this.selectedTypes.length === 0
    ) {
      setSelectedTypes(['All']);
    } else {
      setSelectedTypes(selectedTypes.filter(e => e !== "All"))
    }

  }

  /**
   * 6) & 7) "Clear" (formerly “Reset”) should restore defaults and show all items again.
   *     After clearing, we immediately reapply filters (which are now “empty”),
   *     causing the list to reset.
   */
  resetFilters() {
    this.searchQuery = '';
    this.selectedTypes = ['All'];
    this.dateFrom = null;
    this.dateTo = null;
    this.sortOrder = 'asc';
    this.selectedPageSize = '18';
    this.currentPage = 0;
    this.updatePaginatedItems();

    // Force the componentList to show “everything”:
    //   - an empty search string
    //   - tagQuery = null (because selectedTypes=['All'])
    //   - no date‐range
    //   - sort by name ascending
    this.componentList?.search('');
    this.componentList?.filterByTags(null);
    this.componentList?.filterByDate(undefined, undefined);
    this.componentList?.sort([{ field: 'name', order: this.sortOrder }]);
  }

  /** Called by the “Name” toggle group */
  onSortOrderChange(order: 'asc' | 'desc') {
    this.sortOrder = order;
    this.componentList?.sort([{ field: 'name', order }]);
  }

  /** Called by the “Apply” button */
  applyFilters() {
    // 1) Search
    const trimmed = this.searchQuery.trim();
    if (trimmed) {
      this.componentList?.search(trimmed);
    } else {
      // If it's blank, pass '' so that the search filter resets
      this.componentList?.search('');
    }

    // 2) Tags
    let tagQuery: TagQueryEntry[] | null = null;
    if (!(this.selectedTypes.length === 1 && this.selectedTypes[0] === 'All')) {
      tagQuery = this.selectedTypes.map((typeLabel) => ({
        name: typeLabel,
        type: 'componentType',
      }));
    }
    this.componentList?.filterByTags(tagQuery);

    // 3) Date
    if (this.dateFrom || this.dateTo) {
      this.componentList?.filterByDate(
        this.dateFrom ?? undefined,
        this.dateTo ?? undefined
      );
    } else {
      // If neither date is present, clear any prior date filter:
      this.componentList?.filterByDate(undefined, undefined);
    }

    // 4) Sort
    this.componentList?.sort([{ field: 'name', order: this.sortOrder }]);
  }
}