import { Component, OnInit, inject } from '@angular/core';
import { MatDialog } from '@angular/material/dialog';
import { DialogNewTicketComponent } from 'src/app/components/dialog-new-ticket/dialog-new-ticket.component';
import { AuthService } from 'src/app/auth/auth.service';
import { Router } from '@angular/router';
import { FormBuilder, FormGroup, Validators } from '@angular/forms';
import { DOCUMENT, ViewportScroller } from '@angular/common';
import { Observable, fromEvent, map } from 'rxjs';

@Component({
  selector: 'app-helpdesk-page',
  templateUrl: './helpdesk-page.component.html',
  styleUrls: ['./helpdesk-page.component.scss']
})
export class HelpdeskPageComponent implements OnInit {
  panelOpenState1 = false;
  panelOpenState2 = false;
  panelOpenState3 = false;
  panelOpenState4 = false;
  panelOpenState5 = false;

  faqCategories = [
    {
      id: 'general',
      title: 'General',
      questions: [
        { id: 1,  question: 'What is FAME?',  answer: 'The FAME Federated Marketplace is a Data Space, according to the DSSC definition. FAME provides a technology platform, the Federated Data Asset Marketplace, customized for buying and selling federated data assets to support Embedded Finance application and the financial sector. In addition to this marketplace, FAME offers a Federative Business/Governance model to ensure sovereignty and the sharing and exchange of trusted data assets resources within the digital ecosystems, all grounded on commonly agreed principles.' },
        { id: 19, question: 'I am a data asset producer, how can my data be indexed by FAME?', answer: 'The actual implementation provides a comprehensive procedure (i.e. forms) to allow data providers to ingest their asset into the marketplace. However, APIs are available to make this process fully automatic and machine 2 machine M2M.' },
        { id: 20, question: 'I am a data asset consumer, how can I find relevant data?', answer: 'FAME provides search mechanisms that supports ontologies in particular for the Financial sector (FIBO/FIGI).' },
        { id: 8,  question: 'What kind of assets does the FAME Marketplace support?',  answer: 'FAME offers diverse federated data assets, including classical datasets, value-added assets like AI/XAI models, analytics, algorithms, executable services, and educational content.' },
        { id: 18, question: 'What are the FAME policies? How my data/metadata will be managed?', answer: 'The Assets Policy Manager (APM) in FAME serves two critical roles: managing access policies for assets and providing users with a consolidated view of their accessible assets. Acting as a central authority, it ensures only authorized individuals can access specific assets and offers users transparency and control over their asset portfolios. Overall, the APM strengthens security, promotes efficiency, and fosters better asset management practices in FAME.' },
      ]
    },
    {
      id: 'target-audience',
      title: 'Target Audience',
      questions: [
        { id: 5,  question: 'Is FAME only for B2B? Why not B2C, P2P?',  answer: 'The FAME EU funded project aims to deliver the full-fledged technology solutions to demonstrate a FAME B2B platform, based on a Marketplace and state of the art data driven technologies. FAME can support also B2C and P2P models, but they are considered beyond the scope of the project.' },
        { id: 6,  question: 'What is the target audience of FAME?',  answer: 'FAME focuses on business organizations such as banks, insurance companies, and service providers that integrate financial services into their applications. It offers tailored solutions to help them establish a compliant data supply chain in line with evolving European regulations and legal frameworks, whether in development or already adopted.' },
        { id: 16, question: 'I am a researcher, private citizen, can I use FAME? ', answer: 'FAME offers open search and discovery facilities to anonymous users. However, you will not be able to enroll or trade any asset.' },
        { id: 17, question: 'I am a private citizen how can I federate to FAME?', answer: 'Private citizens, like students, will not be able to federate to fame to trade assets. This goes beyond the scope of the project. However, anybody can browse the catalogue and the public assets, especially the training material which will be offered.' },
      ]
    },
    {
      id: 'federation',
      title: 'Federation',
      questions: [
        { id: 10, question: 'I am a Business Organization, Data Marketplace/Data Space, how can I federate in FAME?', answer: 'FAME implements a federated governance model to ensure trustworthy, sovereign, private, and secure data transactions in line with EU regulations, fostering transparency and consistency. FAME decouples the technology platform of the Marketplace from the FAME Federation and Governance application (FFGA). The FFGA is managed by an on-line application which provides the legal framework and the governance rules of the FAME federated marketplace.' },
        { id: 11, question: 'What is the FAME Federation? What are the rules?', answer: 'The FAME Federated Marketplace enforces the collaboration of trusted, certified organizations that adhere and are compliant to the principles and rules of FAME.' },
        { id: 13, question: 'What is the federation enrolment process?', answer: 'The FFGA supports a process of onboarding organization which is described below: \n<ol><li>ACME [Business organization (legal entity)] wants to enroll either invited or self-start</li><li>ACME has already or self-provides a decentralized identifier DID</li><li>ACME confirm the acceptance of policy rules of the FAME Federation and applies</li><li>A backoffice semi-automatic procedure assess the credentials of ACME</li><li>A consensus mechanism among other federated users will be triggered</li><li>If the majority of the federated users is in favor the new entity is enrolled, otherwise not</li><li> ACME receives a notification on portal of the status of enrolment</li></ol>\nHowever, the participation of a federated member can be revoked, following an internal to FFG procedure. Any federated member, can ask a revoke procedure of any other member presenting the case and asking for a recount of the consensus.  A consensus mechanism among other federated users will be triggered. If the majority of the federated users is in favor the new entity is banned.\n\nThese mechanisms are enabled by DLT technologies (e.g smart contracts for consensus).' },
        { id: 14, question: 'How identities from the FAME federation to the FAME Marketplace are managed?', answer: 'FAME uses decentralized or self-sovereign identities in the form of Decentralized Identifiers (DIDs). Organizations supply Verifiable Credentials (VCs) to users. The VCs digitally replicate traditional credentials like driver&apos;s licenses or university degrees, embedding essential details such as issuer information and specific attributes. These digital credentials are tamper-evident and provide cryptographically verifiable authorship, revolutionizing how personal qualifications and authorizations are managed and authenticated in the digital domain.' },
      ]
    },
    {
      id: 'transactions',
      title: 'Transactions',
      questions: [
        { id: 15, question: 'FAME Marketplace does know the identity of users making transactions?', answer: 'No, it doesn’t. The FAME marketplace manages users via their DID/VC stored in their wallet which allows all the interactions with the marketplace and in particular the buy/sell mechanisms.' },
        { id: 21, question: 'How FAME manages money for the transactions?', answer: 'The actual implementation of the marketplace, uses tokens, which are the internal form of money managed by FAME (FAME Token). Actually tokens buy/cash-out are managed by administrators/automatic procedures outside the scope of the project. The exchange of fiat money and token is intermediated by FAME administrator.<br>The FAME consortium is exploring to adopt the Digital Euro as the FAME token, when it will be deployed by ECB.' },
        { id: 22, question: 'How does the exchange of an asset, and money, take place between seller and buyer?', answer: 'A buyer wishing to purchase an asset must:<br><ol><li>Make a monetary transfer (bank transfer) to FAME (purchases FAME tokens)</li><li>Upon verification of the successful completion of the transfer, the FAME administrator assigns the FAME tokens to the buyer.</li><li>When the buyer purchases an asset on the FAME marketplace the tokens are transferred from the buyer to the seller and the transaction is registered on the blockchain infrastructure</li><li>As result of successful transaction an NFT is generated and assigned to the buyer as proof of purchase</li></ol>' },
        { id: 23, question: 'How does the buyer access the asset?', answer: 'The seller must have previously equipped access to the asset with a protection system that verifies the NFT<br><ol><li>When the buyer wants to access the asset (e.g., download via a link), the buyer&apos;s system communicates the NFT to the producer&apos;s system as proof of purchase.</li></ol>' },
        {
          id: 24,
          question: 'How do the seller cash out her money?',
          answer: `The seller has previously agreed, among other rules they subscribe to, that the FAME administrator can have access to their tokens:
        <ol>
          <li>The seller requests the FAME administrator to transfer the funds in her bank account.</li>
          <li>The FAME administrator transfers the funds to the seller (e.g., bank transfer).</li>
          <li>The administrator burns the seller's tokens.</li>
        </ol>`
        },
        { id: 26, question: 'I am a data producer/consumer. Can my transactions and records belonging to my asset trades be canceled?', answer: 'The short answer is no, as they are recorded in an immutable blockchain. However, these transactions stores only DID.' },
      ]
    }
  ];

  questionForm: FormGroup;

  // Track the currently active category
  activeCategoryId: string | null = null;

  showMailchimp: boolean = false;

  toggleMailchimp(): void {
    this.showMailchimp = !this.showMailchimp;
  }

  constructor(
    public dialog: MatDialog,
    private authService: AuthService,
    private router: Router,
    private fb: FormBuilder
  ) {
    this.questionForm = this.fb.group({
      name: ['', Validators.required],
      email: ['', [Validators.required, Validators.email]],
      subject: ['', Validators.required],
      message: ['', Validators.required],
      privacyPolicy: [false, Validators.requiredTrue]
    });
  }

  openDialog(): void {
      // User is not logged in, redirect to login page with return URL to helpdesk
      this.router.navigate(['/under-construction']);
  }

  onSubmit(): void {
    if (this.questionForm.valid) {
      console.log('Form data:', this.questionForm.value);
      // Call your sendEmail logic only when valid
      this.sendEmail();
      // Optionally, reset the form or provide success feedback here.
    } else {
      // Mark all fields as touched to show validation errors
      this.questionForm.markAllAsTouched();
    }
  }

  /**
   * Scroll to the category section in the main content.
   */
  scrollToCategory(categoryId: string) {
    this.activeCategoryId = categoryId;

    const element = document.getElementById(categoryId);
    if (element) {
      element.scrollIntoView({ behavior: 'smooth' });
    }
  }

  ngOnInit(){}

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  sendEmail(): void {
    window.location.href = 'mailto:support@fame-horizon.eu?subject=Your%20Question';
  }
}