import { DOCUMENT, ViewportScroller } from '@angular/common';
import { HttpClient } from '@angular/common/http';
import { AfterViewInit, Component, ElementRef, OnInit, ViewChild, inject, Renderer2 } from '@angular/core';
import { MatSnackBar } from '@angular/material/snack-bar';
import { Router } from '@angular/router';
import { Observable, Subscription, fromEvent, map, of, shareReplay } from 'rxjs';
import { AuthService } from 'src/app/auth/auth.service';
import { StatsRespOfferingDto } from 'src/app/services/api/dtos/offering.dto';
import { StatsRespDto } from 'src/app/services/api/dtos/stats.resp.dto';
import { GovernanceApiService } from 'src/app/services/api/governance.api.service';
import { ProvenanceApiService } from 'src/app/services/api/provenance.api.service';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { OpenAPIService } from 'src/app/services/open-api.service';
import { environment } from 'src/environments/environment';

// Define an interface for the API response
interface FederationStats {
  federated_organizations_count: number;
  marketplace_users_count: number;
  organizations: any[];
}

@Component({
  selector: 'app-home-page',
  templateUrl: './home-page.component.html',
  styleUrls: ['./home-page.component.scss']
})
export class HomePageComponent implements OnInit, AfterViewInit {


  searchQuery: string = '';
  showOriginalLabel = true;
  showAnimatedLabel = false;
  // Dynamic numbers for each card
  federatedEntities: number = 24;
  onboardedMembers: number = 43;
  publishedAssets: number = 15;
  performedTrades: number = 15;
  trainingMaterialEntries: number = 12;
  publishedCourses: number = 5;

  constructor(private authService: AuthService, private router: Router, private snackBar: MatSnackBar, private renderer: Renderer2, private http: HttpClient, private openAPIService: OpenAPIService, private govApi: GovernanceApiService, private ptApi: ProvenanceApiService, private tmApi: TmApiService) { }

  private animationInterval: any;
  showMailchimp: boolean = false;
  federatedOrganizationsCount?: number;
  marketplaceUsersCount?: number;
  @ViewChild('benefitsSection', { static: true }) benefitsSection!: ElementRef<HTMLElement>;
  private hasAnimated = false;

  // These four calls fire immediately when the component is constructed:
  publishedAssets2 = this.openAPIService.getCount('components');
  publishedCourses2 = this.openAPIService.getCount('trainingResources');
  dataSetCount = this.openAPIService.getCount('components', ['dataSet']);
  softwareCount = this.openAPIService.getCount('components', ['software']);
  libraryCount = this.openAPIService.getCount('components',['library']);
  modelCount = this.openAPIService.getCount('components',['model']);
  serviceCount = this.openAPIService.getCount('components',['service']);
  multimediaCount = this.openAPIService.getCount('components', ['multimedia']);
  documentCount = this.openAPIService.getCount('components', ['document']);
  otherCount = this.openAPIService.getCount('components', ['Web - API', 'Client']);

  toggleMailchimp(): void {
    this.showMailchimp = !this.showMailchimp;
  }

  stats?: StatsRespDto;
  offeringStats?: StatsRespOfferingDto;
  tradesCount?: number;
  private sub = new Subscription();
  //  Define the dummy values you want to show
  private dummyStats: StatsRespDto = {
    members: 27,
    accounts: 8,
    localUsers: 47
  };
  private dummyOfferingStat: StatsRespOfferingDto = {
    offerings: 10
  };

  ngOnInit(): void {
    this.startCycle(); // Begin the repeated 15s animation cycle
    // this.getFederationStats();
    console.log(environment.federationStatsEndpoint);
    // if user not logged in, just wrap the dummy in an Observable
    const stats$ = this.authService.isLoggedIn()
      ? this.govApi.getStats()
      : of(this.dummyStats);

    this.sub.add(
      stats$.subscribe(
        data => {
          console.log('Stats data:', data); // Log the data to console
          this.stats = data;
        },
        err => console.error('Failed to load stats', err)
      )
    );

    // Provenance Stats
    const offeringStats$ = this.authService.isLoggedIn()
      ? this.ptApi.getStats()
      : of(this.dummyOfferingStat);
    this.sub.add(
      offeringStats$.subscribe(
        ptData => {
          console.log('Provenance Stats data:', ptData);
          this.offeringStats = ptData;
          // Optionally store in a separate property, e.g., this.ptStats = ptData;
        },
        err => console.error('Failed to load provenance stats', err)
      )
    );

    // Trading History Stats - Trades Count
    this.sub.add(
      this.tmApi.getTradesCount().subscribe(
        tradesData => {
          console.log('Trades count data:', tradesData);
          this.tradesCount = tradesData;
        },
        err => console.error('Failed to load trades count', err)
      )
    );
  }

  // --- startCycle still in ngOnInit if you need it ---
  // property-initializer version of your stats call:
  private stats$: Observable<FederationStats> = this.http
    .get<FederationStats>(environment.federationStatsEndpoint)
    .pipe(
      // avoid re-requesting on multiple async-pipe subscriptions
      shareReplay(1)
    );

  // derive two separate streams for each count
  federatedOrganizationsCount$ = this.stats$.pipe(
    map(s => s.federated_organizations_count)
  );
  marketplaceUsersCount$ = this.stats$.pipe(
    map(s => s.marketplace_users_count)
  );

  ngAfterViewInit(): void {
    // Load the Sienna script dynamically
    const siennaScript = this.renderer.createElement('script');
    siennaScript.src = 'https://website-widgets.pages.dev/dist/sienna.min.js';
    siennaScript.async = true;
    siennaScript.onload = () => {
      console.log('Sienna script loaded!');
      // Optionally call its global function if available:
      // window['Sienna']?.init(...);
    };
    this.renderer.appendChild(document.body, siennaScript);

    // Wait 1 second before loading the Mailchimp script so the form is present
    setTimeout(() => {
      const mailchimpScript = this.renderer.createElement('script');
      mailchimpScript.type = 'text/javascript';
      mailchimpScript.src = '//s3.amazonaws.com/downloads.mailchimp.com/js/mc-validate.js';
      mailchimpScript.async = true;
      mailchimpScript.onload = () => {
        // Initialize Mailchimp validation once the script is loaded
        (function ($) {
          const form = document.getElementById('mc-embedded-subscribe-form');
          if (form && form.getAttribute('action')) {
            //@ts-ignore
            window.fnames = [];
            //@ts-ignore
            window.ftypes = [];
            //@ts-ignore
            fnames[0] = "EMAIL";
            //@ts-ignore
            ftypes[0] = "email";
            //@ts-ignore
            fnames[1] = "FNAME";
            //@ts-ignore
            ftypes[1] = "text";
            //@ts-ignore
            fnames[2] = "MMERGE2";
            //@ts-ignore
            ftypes[2] = "text";
            //@ts-ignore
            fnames[5] = "MMERGE5";
            //@ts-ignore
            ftypes[5] = "dropdown";
            //@ts-ignore
            fnames[3] = "ADDRESS";
            //@ts-ignore
            ftypes[3] = "address";
            //@ts-ignore
            fnames[4] = "PHONE";
            //@ts-ignore
            ftypes[4] = "phone";
            //@ts-ignore
            console.log('Mailchimp script initialized.');
          } else {
            console.error('Form element or action attribute not found.');
          }
          //@ts-ignore
        })(jQuery);
      };
      this.renderer.appendChild(document.body, mailchimpScript);
    }, 3000); // Delay of 1000 milliseconds

    const obs = new IntersectionObserver(entries => {
      for (let e of entries) {
        if (e.isIntersecting && !this.hasAnimated) {
          this.hasAnimated = true;
          const cards: NodeListOf<HTMLElement> =
            this.benefitsSection.nativeElement.querySelectorAll('.benefit-card');
          cards.forEach((card, idx) => {
            setTimeout(() => card.classList.add('animate-teeter'), idx * 100);
          });
          obs.unobserve(this.benefitsSection.nativeElement);
        }
      }
    }, { threshold: 0.5 });

    obs.observe(this.benefitsSection.nativeElement);
  }

  // Hold timeout IDs for the animation cycle
  private animationTimeout1: any;
  private animationTimeout2: any;
  /**
   * Recursively runs one cycle of the typewriter animation.
   * Cycle timing:
   * 0s: "Search" label is shown.
   * 1s: Switch to animated label ("Embedded finance ML models").
   * 7s: Revert back to "Search", then restart the cycle immediately.
   */
  private startCycle(): void {
    // Only run the cycle if searchQuery is empty
    if (this.searchQuery) {
      this.showAnimatedLabel = false;
      this.showOriginalLabel = true;
      return;
    }

    // Start with the original label
    this.showOriginalLabel = true;
    this.showAnimatedLabel = false;

    // After 1 second, switch to the animated label
    this.animationTimeout1 = setTimeout(() => {
      if (!this.searchQuery) {
        this.showOriginalLabel = false;
        this.showAnimatedLabel = true;
      }
    }, 1000);

    // After 7 seconds (from cycle start), revert back and restart cycle
    this.animationTimeout2 = setTimeout(() => {
      if (!this.searchQuery) {
        this.showAnimatedLabel = false;
        this.showOriginalLabel = true;
        this.startCycle(); // Immediately start the next cycle
      }
    }, 7000);
  }

  /**
   * Stops the animation cycle by clearing the timeouts.
   */
  private stopCycle(): void {
    if (this.animationTimeout1) {
      clearTimeout(this.animationTimeout1);
      this.animationTimeout1 = null;
    }
    if (this.animationTimeout2) {
      clearTimeout(this.animationTimeout2);
      this.animationTimeout2 = null;
    }
    this.showAnimatedLabel = false;
    this.showOriginalLabel = true;
  }

  // Method to navigate based on login status
  checkLoginAndNavigateToAdvancedSearch() {
    // const token = this.authService.getToken();
    // if (token) {
    this.router.navigate(['/search']); // Navigate to the Advanced Search page
    // } else {
    //   this.router.navigate(['/login']);  // Redirect to the login page if not logged in
    // }
  }

  onSearch() {
    if (!this.searchQuery || this.searchQuery.trim() === '') {
      // Show a snack bar message if the search field is empty
      this.snackBar.open('Please enter a search query.', 'Close', { duration: 3000 });
      return;  // Stop the navigation if the search field is empty
    }
    // Always navigate to /search with the search query
    this.router.navigate(['/fdac'], { queryParams: { query: this.searchQuery } });
  }

  /**
   * Called when the input field gains focus.
   * Stops the animation cycle immediately.
   */
  onFocus(): void {
    this.stopCycle();
  }

  onKeyDown(event: KeyboardEvent) {
    if (event.key === 'Enter') {
      this.onSearch();
    }
    if (this.searchQuery && (this.animationTimeout1 || this.animationTimeout2)) {
      this.stopCycle();
    }
  }

  /**
   * If the user leaves (blur) and the field is empty, restart the animation cycle.
   */
  onBlur(): void {
    if (!this.searchQuery) {
      this.startCycle();
    }
  }

  checkLoginAndNavigate() {
    const token = this.authService.getToken();
    if (token) {
      // User is logged in, navigate to the marketplace
      this.router.navigate(['/fdac']);
    } else {
      // User is not logged in, redirect to the login page
      // change from /login to /fdac
      this.router.navigate(['/fdac']);
    }
  }

  // Method for checking login and navigating to Learning Centre
  checkLoginAndNavigateToLearningCentre() {
    const token = this.authService.getToken();
    if (token) {
      this.router.navigate(['/learning-centre']);  // Navigate to Learning Centre
    } else {
      this.router.navigate(['/login']); // Redirect to login if not logged in
    }
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  openYoutube(): void {
    window.open('https://www.youtube.com/watch?v=nHdRubqsPH0', '_blank');
  }
}