import {
  AfterViewInit,
  Component,
  ElementRef,
  HostListener,
  OnInit,
  ViewChild,
} from '@angular/core';
import { TrainingList } from 'src/app/components/training-list/training-list.component';

interface Item {
  title: string;
}

@Component({
  selector: 'app-learning-centre',
  templateUrl: './learning-centre.component.html',
  styleUrls: ['./learning-centre.component.scss'],
})
export class LearningCentreComponent implements OnInit, AfterViewInit {
  loading = true;
  // Items and Pagination
  items: Item[] = Array.from({ length: 100 }, (_, i) => ({
    title: `AI based detection of fraudulent immediate loan requests ${i + 1}`,
  }));
  paginatedItems: Item[] = [];
  selectedPageSize = '18'; // Default value for Items per page
  currentPage = 0;

  // Search and Filters
  searchQuery = ''; // Search query
  selectedTypes = ['All']; // Default selected types
  types = [
    'All',
    'Courses',
    'How-to guides',
    'Webinars',
    'Hands-on examples',
    'Publications',
    'External Resources',
  ];
  dateFrom: Date | null = null; // Date from
  dateTo: Date | null = null; // Date to

  // Dropdowns
  dropdownOpen = false;
  selectedOption = 'Newest';
  options = ['Newest', 'Oldest', 'Relevance'];

  dropdownUserOpen = false;
  selectedUserOption = '18'; // Default value for Items per page dropdown
  optionsUser = ['10', '18', '30'];

  // Filters Sidebar State
  filtersOpen: boolean = false; // To manage filter visibility
  trainingList: TrainingList | null = null;
  @ViewChild('filters') filters!: ElementRef; // Reference to the filters section

  constructor() { }

  sampleTrainingListMethods() {
    this.trainingList?.filterByDate(new Date("2021"), new Date("2026"))
    this.trainingList?.sort([{ field: "title", order: "desc" }])
  }

  @ViewChild("trainingList") set setComponentList(el: TrainingList) {
    this.trainingList = el;
  };


  ngOnInit(): void {
    this.updatePaginatedItems();
    // show spinner for exactly 2 seconds
    setTimeout(() => {
      this.loading = false;
    }, 3000);
  }

  ngAfterViewInit() { }

  /**
   * Toggles the visibility of the Sort dropdown.
   */
  toggleDropdown() {
    this.dropdownOpen = !this.dropdownOpen;
  }

  /**
   * Selects an option from the Sort dropdown.
   * @param option - The selected sort option.
   */
  selectOption(option: string) {
    this.selectedOption = option;
    this.dropdownOpen = false;
  }

  /**
   * Toggles the visibility of the Items Per Page dropdown.
   */
  toggleUserDropdown() {
    this.dropdownUserOpen = !this.dropdownUserOpen;
  }

  /**
   * Selects an option from the Items Per Page dropdown.
   * @param option - The selected items per page option.
   */
  selectUserOption(option: string) {
    this.selectedUserOption = option;
    this.onPageSizeChange(option);
    this.dropdownUserOpen = false;
  }

  /**
   * HostListener to handle clicks outside of dropdowns and filters.
   * Closes dropdowns and filters when clicking outside.
   */
  @HostListener('document:click', ['$event'])
  handleDocumentClick(event: Event) {
    const target = event.target as HTMLElement;
    const clickedInsideSort = target.closest('.dropdown.sort');
    const clickedInsideUser = target.closest('.dropdown.user');
    const clickedInsideFilters = this.filters.nativeElement.contains(target);
    const clickedFilterToggle = target.closest('.filter-toggle-btn');

    if (!clickedInsideSort) {
      this.dropdownOpen = false;
    }

    if (!clickedInsideUser) {
      this.dropdownUserOpen = false;
    }

    // Close filters if clicked outside and not on the toggle button
    if (this.filtersOpen && !clickedInsideFilters && !clickedFilterToggle) {
      this.closeFilters();
    }
  }

  /**
   * Updates the list of items displayed based on pagination.
   */
  updatePaginatedItems() {
    const pageSize = parseInt(this.selectedPageSize, 10); // Convert to number
    const startIndex = this.currentPage * pageSize;
    const endIndex = startIndex + pageSize;
    this.paginatedItems = this.items.slice(startIndex, endIndex);
  }

  /**
   * Handles changes in the page size selection.
   * @param newSize - The new page size selected by the user.
   */
  onPageSizeChange(newSize: string) {
    this.selectedPageSize = newSize;
    this.currentPage = 0; // Reset to first page
    this.updatePaginatedItems();
  }

  /**
   * Handles page change events from the paginator.
   * @param event - The paginator event containing the new page index.
   */
  onPageChange(event: any) {
    this.currentPage = event.pageIndex;
    this.updatePaginatedItems();
  }

  /**
   * Handles changes in the selected types for filtering.
   * @param selectedTypes - The array of types selected by the user.
   */
  onTypeChange(selectedTypes: string[]) {
    const allCombinations = [
      ['Courses', 'How-to guides', 'Webinars'],
      ['Courses', 'How-to guides'],
      ['Courses', 'Hands-on examples'],
      ['Courses', 'Publications'],
      ['Courses', 'External Resources'],
      ['How-to guides', 'Webinars'],
      ['How-to guides', 'Hands-on examples'],
      ['How-to guides', 'Publications'],
      ['How-to guides', 'External Resources'],
      ['Webinars', 'Hands-on examples'],
      ['Webinars', 'Publications'],
      ['Webinars', 'External Resources'],
      ['Hands-on examples', 'Publications'],
      ['Hands-on examples', 'External Resources'],
      ['Publications', 'External Resources'],
      ['Courses', 'How-to guides', 'Webinars', 'Hands-on examples'],
      ['Courses', 'How-to guides', 'Webinars', 'Publications'],
      [
        'Courses',
        'How-to guides',
        'Webinars',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Hands-on examples',
        'Publications',
      ],
      [
        'Courses',
        'How-to guides',
        'Hands-on examples',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'Webinars',
        'Hands-on examples',
        'Publications',
      ],
      [
        'Courses',
        'Webinars',
        'Hands-on examples',
        'External Resources',
      ],
      [
        'Courses',
        'Webinars',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'Publications',
      ],
      [
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'External Resources',
      ],
      [
        'How-to guides',
        'Webinars',
        'Publications',
        'External Resources',
      ],
      [
        'How-to guides',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'Webinars',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'Publications',
      ],
      [
        'Courses',
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Webinars',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'Webinars',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
      [
        'Courses',
        'How-to guides',
        'Webinars',
        'Hands-on examples',
        'Publications',
        'External Resources',
      ],
    ];

    const setSelectedTypes = (types: string[]) => {
      this.selectedTypes = types;
      console.log(this.selectedTypes);
    };

    const shouldSelectAll = (types: string[]): boolean => {
      return allCombinations.some(
        (combination) =>
          combination.every((type) => types.includes(type)) &&
          combination.length === types.length
      );
    };

    if (selectedTypes.includes('All') && selectedTypes.length > 1) {
      const filteredTypes = selectedTypes.filter((type) => type !== 'All');
      if (shouldSelectAll(filteredTypes)) {
        setSelectedTypes(['All']);
      } else {
        setSelectedTypes(filteredTypes);
      }
      console.log(this.selectedTypes + '1');
    } else if (selectedTypes.length === 0) {
      setSelectedTypes(['All']);
      console.log(this.selectedTypes + '2');
    } else {
      setSelectedTypes(selectedTypes);
      console.log(this.selectedTypes + '3');
    }
  }

  /**
   * Resets all filters to their default values.
   */
  resetFilters() {
    this.searchQuery = '';
    this.selectedTypes = ['All'];
    this.dateFrom = null;
    this.dateTo = null;
    this.selectedPageSize = '18';
    this.selectedUserOption = '18';
    this.currentPage = 0;
    this.sortOrder = 'asc';
  
    this.updatePaginatedItems();

    this.trainingList?.search('');
    this.trainingList?.filterByDate(undefined, undefined);
    this.trainingList?.sort([{ field: 'title', order: this.sortOrder }]);
  }
  
  /**
   * Applies the selected filters.
   */
  applyFilters() {
    // Handle filter application logic here
    // For example, emit an event or call a service to fetch filtered data
    console.log('Filters Applied:', {
      searchQuery: this.searchQuery,
      selectedTypes: this.selectedTypes,
      dateFrom: this.dateFrom,
      dateTo: this.dateTo,
      // selectedSort: this.selectedSort,
      selectedPageSize: this.selectedPageSize,
    });

    // Implement the actual filter logic, such as fetching filtered data
    // 1. Search
    this.trainingList?.search(this.searchQuery.trim());

    // 3. Date
    if (this.dateFrom || this.dateTo) {
      this.trainingList?.filterByDate(this.dateFrom ?? undefined, this.dateTo ?? undefined);
    }

    // 4. Sort
    this.trainingList?.sort([{ field: 'name', order: this.sortOrder }]);
  }

  /**
   * Toggles the visibility of the filters sidebar.
   */
  toggleFilters() {
    this.filtersOpen = !this.filtersOpen;
    if (this.filtersOpen) {
      document.body.style.overflow = 'hidden'; // Prevent background scrolling
    } else {
      document.body.style.overflow = 'auto';
    }
  }

  /**
   * Closes the filters sidebar.
   */
  closeFilters() {
    this.filtersOpen = false;
    document.body.style.overflow = 'auto';
  }

  sortOrder: 'asc' | 'desc' = 'asc';
  onSortOrderChange(order: 'asc' | 'desc') {
    this.sortOrder = order;
    this.trainingList?.sort([{ field: 'title', order }]);   // choose field!
  }
}