import { Component, OnDestroy, OnInit, inject } from '@angular/core';
import {
  CredentialMapper,
  UniformVerifiablePresentation,
} from '@sphereon/ssi-types';
import { AuthService } from 'src/app/auth/auth.service';
import { Router, ActivatedRoute } from '@angular/router';
import { CommonService } from 'src/app/services/common.service';
import { SnackbarService } from 'src/app/services/snackbar/snackbar.service';
import { Observable, Subscription, fromEvent, map } from 'rxjs';
import { DOCUMENT, ViewportScroller } from '@angular/common';

@Component({
  selector: 'app-login',
  templateUrl: './login.component.html',
  styleUrls: ['./login.component.scss'],
})
export class LoginComponent implements OnInit, OnDestroy {

  private refinedSub: Subscription | undefined;
  
  constructor(
    private commonService: CommonService,
    private authService: AuthService,
    private router: Router,
    private route: ActivatedRoute,
    private snackbarService: SnackbarService,
  ) { }

  ngOnInit(): void {
    this.refinedSub = this.commonService.refinedEvent.subscribe(value => {
      this.wrapperPresentation(value);
    });
  }

  ngOnDestroy(): void {
    if (this.refinedSub) {
      this.refinedSub.unsubscribe();
    }
  }

  cancel() {
    this.router.navigate(['/']); // Redirect to home page
  }

  wrapperPresentation(payload: any) {
    try {
      const wrappedPresentation =
        CredentialMapper.toWrappedVerifiablePresentation(
          Array.isArray(payload.vp_token)
            ? payload.vp_token[0]
            : payload.vp_token!
        );
      const vpToken = Array.isArray(payload.vp_token)
        ? payload.vp_token[0]
        : payload.vp_token!;

      this.authService.generateJWTToken(vpToken).subscribe(
        (response) => {
          console.log('Login successfully');
          // console.log('Server response:', response);
          this.authService.setToken(response.jwtToken);
          // console.log('Session storage item:', sessionStorage.getItem('jwt'));

          // THEN, let the nav bar know
          this.commonService.popCloseEvent.emit({ firstName: '...', lastName: '...' });

          // const user = this.authService.getUserDetails(); 
          // console.log('User from token:', user);
        },
        (error) => {
          console.error('Error generating token:', error);
          // Check for error status 401
        if (error.status === 401) {
          this.snackbarService.error('Login failed. Please try again.', 'Close');
        } else {
          this.snackbarService.error('An unexpected error occurred. Please try again later.', 'Close');
        }
        }
      );
      const presentation =
        wrappedPresentation?.presentation as UniformVerifiablePresentation;
      const subjects =
        presentation?.verifiableCredential[0].credential.credentialSubject;
      const subject = Array.isArray(subjects) ? subjects[0] : subjects!;
      const proofs = presentation?.verifiableCredential[0].credential.proof;
      let jwtPayload = JSON.stringify(proofs);
      const jsonDataPayload = JSON.parse(jwtPayload);
      const jwt = jsonDataPayload.jwt;

      // Set the JWT token in sessionStorage
      //this.authService.setToken(jwt);

      // Emit event for login and store user data
      this.commonService.popCloseEvent.emit(subject);

      // Show a success snackbar
      // this.snackbarService.success('Login successful!', 'Close');

      // Redirect back to the returnUrl if provided, else to the home page
      const returnUrl = this.route.snapshot.queryParams['returnUrl'] || '/';
      this.router.navigate([returnUrl]);
    } catch (error) {
      // Show an error snackbar if login fails
      this.snackbarService.error(
        'Login failed. Please try again.',
        'Close'
      );
    }
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  redirectToAboutUs(event: Event): void {
    event.preventDefault(); // Prevent any default behavior
    // Open /about-us with a query parameter 'stepper=1' in a new tab
    window.open('/about-us?stepper=1', '_blank');
  }
}