import { AfterViewInit, Component, OnDestroy, OnInit, ViewChild, inject, TemplateRef } from '@angular/core';
import { NgForm } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { ActivatedRoute, Router } from '@angular/router';
import { GovernanceApiService } from '../../services/api/governance.api.service';
import { Observable, fromEvent, map, take } from 'rxjs';
import { DOCUMENT, ViewportScroller } from '@angular/common';
import { SnackbarService } from 'src/app/services/snackbar/snackbar.service';
import { MatDialog } from '@angular/material/dialog';
import { MarkdownPreviewService, MarkdownPreviewResult } from 'src/app/services/markdown-preview/markdown-preview.service';
import { MarkdownPreviewModalComponent } from 'src/app/components/markdown-preview-modal/markdown-preview-modal.component';
import { PatDialogComponent } from 'src/app/components/pat-dialog/pat-dialog.component';

@Component({
  selector: 'app-offering-definition',
  templateUrl: './offering-definition.component.html',
  styleUrls: ['./offering-definition.component.scss'],
})
export class OfferingDefinitionComponent implements OnInit, OnDestroy, AfterViewInit {
  private defaultDuration: number = 3000;
  protected txHash: any;
  offeringId: any = '';
  subscribePt: any;
  subscribeReq: any;

  constructor(
    private snackBar: MatSnackBar,
    private provenanceApiService: ProvenanceApiService,
    private governanceApiService: GovernanceApiService,
    private clipboardManager: Clipboard,
    private route: ActivatedRoute,
    private router: Router,
    private snackbarService: SnackbarService,
    private dialog: MatDialog,
    private markdownPreviewService: MarkdownPreviewService
  ) {}

  readonly accessDuration = 'Access duration';
  readonly accessCount = 'Number of accesses';
  readonly contentVolume = 'Volume of content';
  readonly tradeUnits: string[] = [this.accessDuration, this.accessCount, this.contentVolume];
  readonly durationTypes: string[] = ['m', 'H', 'D', 'M', 'Y'];
  readonly volumeTypes: string[] = ['KB', 'MB', 'GB', 'TB'];
  isSubmitted: boolean = false;
  isPat: boolean = false;
  isProcessed: boolean = false;
  asset: any;
  copyToClipboard: any;
  selectedUnit: any;

  @ViewChild('offeringForm') offeringForm!: NgForm;
  offering = {
    asset: '',
    assetName: '',
    title: '',
    price: '',
    summary: '',
    scope: '',
    unit: {
      duration: 1,
      durationType: this.durationTypes[3],
      retrievals: 1,
      volume: 1,
      volumeType: this.volumeTypes[0],
    },
    license: '',
    sla: '',
    tid: '',
    cdsinst: '',
  };

  ngOnInit() {
    this.isProcessed = false;
    this.isSubmitted = false;
    const offeringStored = localStorage.getItem('offering') || '{}';
    if (offeringStored !== '{}') this.offering = JSON.parse(offeringStored);
    this.route.queryParams.subscribe((params) => {
      console.log('Params', params);
      if (!params['aid'] && !params['price']) return;
      this.offering.asset = params['aid'];
      if (this.offering.asset)
        this.offering.asset = decodeURI(this.offering.asset);
      if (params['title']) this.offering.assetName = decodeURI(params['title']);
      // Note: price handling moved below to ensure PAT suggested price takes precedence
      
      // Restore complete form state from PAT navigation
      if (params['selectedUnit'] || params['offeringTitle'] || params['summary'] || params['formPrice'] || params['scope'] || params['license'] || params['sla'] || params['tid'] || params['cdsinst']) {
        
        // Restore unit type selection and values
        if (params['selectedUnit']) {
          this.selectedUnit = params['selectedUnit'];
          
          if (params['duration'] && params['durationType']) {
            this.offering.unit.duration = Number(params['duration']);
            this.offering.unit.durationType = params['durationType'];
          }
          
          if (params['retrievals']) {
            this.offering.unit.retrievals = Number(params['retrievals']);
          }
          
          if (params['volume'] && params['volumeType']) {
            this.offering.unit.volume = Number(params['volume']);
            this.offering.unit.volumeType = params['volumeType'];
          }
        }
        
        // Restore all other form fields
        if (params['offeringTitle']) this.offering.title = params['offeringTitle'];
        if (params['summary']) this.offering.summary = params['summary'];
        
        // Price handling: PAT suggested price takes precedence over form price
        if (params['price']) {
          this.offering.price = params['price']; // PAT suggested price
        } else if (params['formPrice']) {
          this.offering.price = params['formPrice']; // Fallback to preserved form price
        }
        if (params['scope']) this.offering.scope = params['scope'];
        if (params['license']) this.offering.license = params['license'];
        if (params['sla']) this.offering.sla = params['sla'];
        if (params['tid']) this.offering.tid = params['tid'];
        if (params['cdsinst']) this.offering.cdsinst = params['cdsinst'];
        
        // Store correlation ID from PAT for potential notification
        if (params['correlationId']) this.patCorrelationId = params['correlationId'];
        
        console.log('Restored complete form state:', {
          selectedUnit: this.selectedUnit,
          offering: this.offering
        });
      }
    });
  }

  ngAfterViewInit(): void {
    const script = document.createElement('script');
    script.src = 'https://website-widgets.pages.dev/dist/sienna.min.js';
    script.async = true;  // or false, depending on your needs
  
    script.onload = () => {
      console.log('Sienna script loaded!');
      // Optionally, call any global functions provided by the script
      // For example: window['Sienna']?.init(...);
    };
  
    // Append the script element to the document head (or body)
    document.head.appendChild(script);
  }

  ngOnDestroy(): void {
    if (this.subscribePt) this.subscribePt.unsubscribe();
    if (this.subscribeReq) this.subscribeReq.unsubscribe();
  }

  onDiscard() {
    this.offeringForm.resetForm();
  }

  // Simplified, anthropic, and clear messages based on backend info
  backendMessages = {
    offeringTitle: 'Provide a concise offering title. Only Latin letters, numbers, punctuation, and spaces are allowed. Maximum length: 80 characters (Required)',
    summary: 'Provide a brief overview of the offering. Only Latin letters, numbers, punctuation, spaces, and newlines are allowed. Maximum length: 5000 characters (Required)',
    unitType: 'Select the unit type for this offering, depending on whether you want to apply a (i) Subscription, or a (ii) Pay-as-you-use, or a (iii) Pay-as-you-go Business Model',
    unitPrice: 'Enter the unit price in FDE (e.g., 1000.00). Two decimals are required',
    limitations: 'Specify any limitations or scope details (Optional)',
    terms: 'Provide the formal terms and conditions for this offering. Basic Markdown formatting is supported: **bold**, *italic*, # headings, and lists. Maximum length: 30000 characters (Required)',
    sla: 'Provide the service level agreement details. Basic Markdown formatting is supported: **bold**, *italic*, # headings, and lists. Maximum length: 30000 characters (Optional)',
    tradingAccount: 'Provide your trading account address, which can be located in the Accounts section in the Profile page (Required)',
    cdsinst: 'Provide a JSON object for the offering with "name" as a key and "offering name" as a value, following the format: {"name":"My Offering Name"}'
  };

  ONE_LINER_MAXL  = 80;
  TXT_LONG_MAXL   = 5000;
  TXT_SHORT_MAXL  = 500;
  DOCUMENT_MAXL   = 30000;


  // price: up to 6 digits, dot, exactly 2 decimals
  pricePattern    = '^(0|[1-9]\\d{0,5})\\.\\d{2}$';

  // Ethereum address
  ethPattern      = '^0x[a-fA-F0-9]{40}$';

  // very basic JSON literal validator
  jsonPattern     = '^\\s*(\\{.*\\}|\\[.*\\])\\s*$';

  isLoading = false;
  patCorrelationId?: string;


  onSubmit() {
    if (this.offeringForm.valid) {
      const offering = this.formatOffering();
      this.isSubmitted = true;
      this.isLoading   = true;
      console.log('Submitting offering:', offering);
      this.subscribePt = this.provenanceApiService.submitOffering(offering, this.patCorrelationId)
        .subscribe(
          (response) => {
            this.isLoading = false;
            if (!response.id) return;
            this.snackbarService.success(
              'Offering confirmed with OID ' + response.id,
              'Close',
              3000
            );
            this.isProcessed = true;
            this.offeringId = response.id;
            localStorage.removeItem('offering');

            // Reset form
            this.offeringForm.resetForm();
            this.offeringForm.form.disable();
            this.isPat = true;

            // Navigate directly to offering details
            this.router.navigate(['/offering-details'], {
              queryParams: {
                oid: response.id,
                title: this.offering.title
              }
            });
          },
          (error) => {
            this.isLoading = false;
            this.handleValidationError(error);
          }
        );
    } else {
      this.snackBar.open('Please fill all the required fields', 'Info', {
        duration: this.defaultDuration,
      });
    }
  }

  private handleValidationError(error: any): void {
    let errorMessage = 'An error occurred while processing your request.';
    
    if (error.status === 400 && error.error) {
      // Extract class-validator messages
      if (error.error.message && Array.isArray(error.error.message)) {
        // Join validation messages with line breaks
        errorMessage = error.error.message.join('\n');
      } else if (typeof error.error.message === 'string') {
        errorMessage = error.error.message;
      } else if (error.error.details) {
        errorMessage = error.error.details;
      }
    } else if (error.message) {
      errorMessage = error.message;
    }
    
    this.snackbarService.error(errorMessage, 'Close', 5000);
    this.isSubmitted = false; // Allow user to retry
  }

  formatPrice() {
    if (this.offering.price !== undefined && this.offering.price !== null) {
      const parsed = parseFloat(this.offering.price);
      // Validate numeric constraints first
      if (!isNaN(parsed) && parsed >= 0 && parsed <= 1000000) {
        // Reassign the value as a string with exactly two decimals.
        this.offering.price = parsed.toFixed(2);
      }
    }
  }

  formatOffering() {
    // we must remove some unwanted properties and normalize some others
    const offeringToSubmit = JSON.parse(JSON.stringify(this.offering));
    delete offeringToSubmit.assetName;
    delete offeringToSubmit.unit.durationType;
    delete offeringToSubmit.unit.volumeType;
    if (this.selectedUnit === this.accessDuration) {
      offeringToSubmit.unit.duration = this.offering.unit.duration + this.offering.unit.durationType;
      delete offeringToSubmit.unit.retrievals;
      delete offeringToSubmit.unit.volume;
    } else if (this.selectedUnit === this.accessCount) {
      delete offeringToSubmit.unit.duration;
      delete offeringToSubmit.unit.volume;
    } else if (this.selectedUnit === this.contentVolume) {
      offeringToSubmit.unit.volume = this.offering.unit.volume + this.offering.unit.volumeType;
      delete offeringToSubmit.unit.duration;
      delete offeringToSubmit.unit.retrievals;
    }
    for (const key in offeringToSubmit) {
      if (offeringToSubmit.hasOwnProperty(key)) {
        const value = offeringToSubmit[key];
        // Check if the value is null/undefined or if it's a string that is empty or only whitespace.
        if (value == null || (typeof value === 'string' && value.trim() === ''))
          delete offeringToSubmit[key];
      }
    }
      
    console.log('OFFERING TO SUBMIT', offeringToSubmit);
    return offeringToSubmit;
  }

  copyUri(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.offeringId);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  goToFdacOfferingDetails($event: any): void {
    $event.preventDefault();
  
    this.router.navigate(['/offering-details'], {
      queryParams: {
        oid: this.offeringId,
        title: this.offering.title
      }
    });
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    this.subscribeReq = this.governanceApiService
      .checkRequestStatus(this.offeringId)
      .pipe(take(1))
      .subscribe(
        (response: any) => {
          if (!response) return;
          console.log('Response', response);
          this.snackBar.open('Submitted Request:', response.message, {
            duration: 6000,
          });
        },
        (error) => {
          console.log('Error', error);
          this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
          });
        }
      );
  }

  openPat(event: MouseEvent) {
    event.preventDefault();
    
    // Prepare complete form state
    const formState: any = {
      selectedUnit: this.selectedUnit,
      title: this.offering.title,
      summary: this.offering.summary,
      price: this.offering.price,
      scope: this.offering.scope,
      license: this.offering.license,
      sla: this.offering.sla,
      tid: this.offering.tid,
      cdsinst: this.offering.cdsinst
    };

    // Add unit type specific data based on current selection
    if (this.selectedUnit === this.accessDuration) {
      formState.duration = this.offering.unit.duration;
      formState.durationType = this.offering.unit.durationType;
    } else if (this.selectedUnit === this.accessCount) {
      formState.retrievals = this.offering.unit.retrievals;
    } else if (this.selectedUnit === this.contentVolume) {
      formState.volume = this.offering.unit.volume;
      formState.volumeType = this.offering.unit.volumeType;
    }

    this.dialog.open(PatDialogComponent, {
      width: '400px',
      data: {
        assetId: this.offering.asset,
        assetTitle: this.offering.assetName,
        formState: formState
      }
    });
  }

  goToCatalogue($event: any) {
    $event.preventDefault();
    this.router.navigate(['/asset/' + this.offering.asset]);
  }

  get breadcrumbs(): {url: string; label: string }[] {
    return [
      { url: '/fdac', label: 'Assets' },
      { url: '/asset/' + this.offering.asset, label: 'Asset details'},
      { url: '/offering-definition', label: 'Offering definition'},
    ]
  }

  
  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }

  get isUnitTypeValid(): boolean {
    if (!this.selectedUnit) return false;

    if (this.selectedUnit === this.accessDuration) {
      return !!(this.offering.unit.duration && 
               this.offering.unit.duration > 0 && 
               this.offering.unit.durationType);
    } else if (this.selectedUnit === this.accessCount) {
      return !!(this.offering.unit.retrievals && 
               this.offering.unit.retrievals > 0);
    } else if (this.selectedUnit === this.contentVolume) {
      return !!(this.offering.unit.volume && 
               this.offering.unit.volume > 0 && 
               this.offering.unit.volumeType);
    }

    return false;
  }

  /**
   * Handle Markdown preview for the license field
   */
  previewLicense(): void {
    const result: MarkdownPreviewResult = this.markdownPreviewService.previewMarkdown(
      this.offering.license,
      this.DOCUMENT_MAXL
    );

    if (!result.success) {
      // Show toast with appropriate message
      this.snackbarService.info(result.message || 'Preview not available', 'Close', 3000);
      return;
    }

    // Show modal with formatted content
    this.dialog.open(MarkdownPreviewModalComponent, {
      width: '80%',
      maxWidth: '800px',
      maxHeight: '80vh',
      data: {
        html: result.html,
        title: 'Terms and Conditions Preview'
      }
    });
  }

  /**
   * Handle Markdown preview for the SLA field
   */
  previewSla(): void {
    const result: MarkdownPreviewResult = this.markdownPreviewService.previewMarkdown(
      this.offering.sla,
      this.DOCUMENT_MAXL
    );

    if (!result.success) {
      // Show toast with appropriate message
      this.snackbarService.info(result.message || 'Preview not available', 'Close', 3000);
      return;
    }

    // Show modal with formatted content
    this.dialog.open(MarkdownPreviewModalComponent, {
      width: '80%',
      maxWidth: '800px',
      maxHeight: '80vh',
      data: {
        html: result.html,
        title: 'Service Level Agreement Preview'
      }
    });
  }
}