import { Component, OnDestroy, OnInit, ChangeDetectorRef, inject, AfterViewInit } from '@angular/core';
import { MatSnackBar } from '@angular/material/snack-bar';
import { MatDialog } from '@angular/material/dialog';
import { MetamaskChainService } from '../../services/chain/metamask.chain.service';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { Clipboard } from '@angular/cdk/clipboard';
import { ActivatedRoute, Router } from '@angular/router';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { Observable, combineLatest, fromEvent, lastValueFrom, map, take, firstValueFrom } from 'rxjs';
import { GovernanceApiService } from '../../services/api/governance.api.service';
import { PolicyIntegrationService } from '../../services/policy-integration.service';
import { BreadcrumbSimple } from 'src/app/components/breadcrumb-simple/breadcrumb-simple.component';
import { DOCUMENT, ViewportScroller } from '@angular/common';
import { ArchiveOfferingDialogComponent } from '../../components/archive-offering-dialog/archive-offering-dialog.component';
import { ApproveAllowanceDialogComponent } from '../../components/approve-allowance-dialog/approve-allowance-dialog.component';

@Component({
  selector: 'app-offering-details',
  templateUrl: './offering-details.component.html',
  styleUrls: ['./offering-details.component.scss'],
})
export class OfferingDetailstComponent implements OnInit, OnDestroy, AfterViewInit {

  // flags for each Business Model
  isSubscription: boolean = false;
  isPayAsYouUse: boolean = false;
  isPayAsYouGo: boolean = false;

  private connectedAccount: string | undefined;
  private userAllowance = 0;

  private defaultDuration: number = 8000;
  protected txHash: any;
  isSubmitted: any;
  offeringDetailsId: any;
  subscribeTm: any;
  hasEnoughtApproval: boolean = false;
  offeringId = '';
  assetId = '';
  assetName: any = '';
  offering: any;
  offeringStatus: string = '';
  assetDetails: any;
  isAssetWriteable: boolean = false;
  unit: string = '';
  unitAdditional: string = '';
  unitType: string = '';

  constructor(
    private snackBar: MatSnackBar,
    private metamaskService: MetamaskChainService,
    private provenanceApiService: ProvenanceApiService,
    private governanceApiService: GovernanceApiService,
    private tmApiService: TmApiService,
    private router: Router,
    private route: ActivatedRoute,
    private clipboardManager: Clipboard,
    private changeDetectorRef: ChangeDetectorRef,
    private dialog: MatDialog,
    private policyIntegrationService: PolicyIntegrationService
  ) {}

  account$ = this.metamaskService.account$;
  isSupportedChain$ = this.metamaskService.isSupportedChain$;

  isMetamaskConnected: boolean = false;

  durationTypes: string[] = ['m', 'H', 'D', 'M', 'Y'];
  volumeTypes: string[] = ['KB', 'MB', 'GB', 'TB'];

  overviewItems: any = [];

  ngOnInit() {
    this.setOfferingIdFromRoute()
    .then(({ oid }) =>
    // 1) fetch the offering…
    this.setOfferingDetails(oid)
        .then(() => {
        // 2) then fetch the asset by reading this.offering.asset
        const realAid = this.offering.asset;
        return this.setAssetDetails(realAid);
        })
    )
    .then(() => this.setConnectedAccount())
    .then(() => this.setAllowance())
    .then(() => this.setApprovalBasedOnAllowance())
    .then(() => this.checkAsset());
  }

  ngOnDestroy(): void {
    if (this.subscribeTm) this.subscribeTm.unsubscribe();
  }

  ngAfterViewInit(): void {
    const script = document.createElement('script');
    script.src = 'https://website-widgets.pages.dev/dist/sienna.min.js';
    script.async = true;  // or false, depending on your needs
  
    script.onload = () => {
      console.log('Sienna script loaded!');
      // Optionally, call any global functions provided by the script
      // For example: window['Sienna']?.init(...);
    };
  
    // Append the script element to the document head (or body)
    document.head.appendChild(script);
  }

  get isOfferingArchived() {
    return this.offeringStatus === 'ARCHIVED';
  }

  get isOfferingActive() {
    return this.offeringStatus === 'ACTIVE';
  }

  get canArchiveOffering() {
    return this.isOfferingActive && this.isAssetWriteable;
  }

  get disablePurchaseButton() {
    return (
      this.isSubmitted || !this.connectedAccount || this.isOfferingArchived
    );
  }

  private setOfferingIdFromRoute(): Promise<{ oid: string }> {
    return lastValueFrom(
      combineLatest([this.route.params, this.route.queryParams]).pipe(take(1))
    ).then(([params, query]) => {
      this.offeringId = params['oid'] ?? query['oid'] ?? '';
      this.assetName  = query['title'] ? decodeURIComponent(query['title']) : '';
      if (!this.offeringId) {
        this.router.navigate(['/']);
        return { oid: '' };
      }
      return { oid: this.offeringId };
    });
  }
  

  private async setConnectedAccount(): Promise<string> {
    return this.metamaskService.getAccount().then((res) => {
      this.connectedAccount = res;
      return res;
    });
  }

  private async setOfferingDetails(oid: string) {
    return lastValueFrom(this.provenanceApiService.getOfferingDetails(oid))
      .then((data: any) => {
        if (data) {
          this.offering = data.offering; // Extract offering data from OfferingRecord
          this.offeringStatus = data.status; // Extract status from OfferingRecord
          console.log('Offering details:', this.offering, 'Status:', this.offeringStatus);
          this.decodeCap(this.offering);
          const desc = this.getUnitDescription();
          // Clear the array before adding new item to prevent duplication
          this.overviewItems = [];
          this.overviewItems.push({
            title: this.offering.title,
            value: this.offering.price,
            description: desc,
            isPositive: true,
            icon: '../../../assets/icons/ta_accounts.png',
          });
          this.changeDetectorRef.markForCheck();
          return data;
        }
      })
      .catch((error) => {
        console.log('Error', error);
        this.snackBar.open('Error fetching offering details', 'Error', {
          duration: this.defaultDuration,
        });
      });
  }

  /** Fetches the asset details by ID and logs them */
  private async setAssetDetails(aid: string): Promise<any> {
    console.log('⏺️ setAssetDetails – assetId:', aid);
    try {
      const data = await lastValueFrom(
        this.provenanceApiService.getAssetDetails(aid)
      );
      console.log('Asset details:', data);
      this.assetDetails = data;
      return data;
    } catch (err) {
      console.error('Error fetching asset details', err);
      this.snackBar.open('Error fetching asset details', 'Error', {
        duration: this.defaultDuration,
      });
      return null;
    }
  }

  private async setAllowance(): Promise<number> {
    if (!this.connectedAccount) {
      console.error('No connected account');
      return 0;
    }
    return lastValueFrom(
      this.tmApiService.checkApprovalAmount(this.connectedAccount)
    ).then((amount) => {
      this.userAllowance = amount as number;
      return amount;
    });
  }

  private setApprovalBasedOnAllowance() {
    if (this.offering.price > this.userAllowance) {
      this.hasEnoughtApproval = false;
    } else {
      this.hasEnoughtApproval = true;
    }
  }

  private async checkAsset(): Promise<void> {
    try {
      const assetId = this.offering.asset;
      const response = await this.policyIntegrationService.assetContentAccessController.checkOne(assetId);
      const json = response?.json;
      this.isAssetWriteable = json?.hasAccess === true && json?.isActive === true;
    } catch (error) {
      this.isAssetWriteable = false;
      console.log('Error checking status for asset:', this.offering.asset, error);
    }
  }

  isLoading: boolean = false;
  isUnpublishingLoading: boolean = false;
  transactionStatusMessage: string = '';
  async purchaseOffer(): Promise<void> {
  try {
    // Re-check allowance before proceeding
    await this.setAllowance();

    // If allowance is insufficient, show dialog prompting to go to profile page
    if (this.offering.price > this.userAllowance) {
      this.dialog.open(ApproveAllowanceDialogComponent, {
        width: '600px',
        disableClose: false,
        data: {
          currentAllowance: this.userAllowance,
          requiredAmount: this.offering.price,
          offeringTitle: this.offering.title
        }
      });

      // Stop purchase flow - user must go to profile page to set allowance
      return;
    }

    // 1. Get the transaction submission details from the API service.
    const submissionOrderTx = await this.tmApiService.getSubmissionOrderTx(this.offeringId);

    // 2. Sign and send the transaction via MetaMask.
    const txHash1 = await this.metamaskService.signAndSentTransaction(submissionOrderTx);
    this.txHash = txHash1;
    this.isSubmitted = true;
    this.snackBar.open('Transaction sent', 'Info', { duration: this.defaultDuration });
    this.changeDetectorRef.markForCheck();

    // 3. Show the spinner while waiting for the transaction receipt.
    this.isLoading = true;
    const receipt1 = await this.metamaskService.waitForTransactionReceipt(txHash1);
    this.isLoading = false;

    // If the first transaction fails, abort the process.
    if (receipt1.status !== 'success') {
      console.error('Transaction failed:', receipt1);
      this.transactionStatusMessage = 'Transaction failed. Please try again.';
      this.snackBar.open('Transaction failed. Please try again.', 'Close', { duration: 5000 });
      return;
    }

    // ONLY FOR pay-as-you-go business model, the provider must have allowance for withdrawing funds from buyer account. Allowance amount is increased of the offer price. 
    if (this.offering.unit.volume) {
      const unsignedAllowanceTx = await this.tmApiService.inscreaseAllowanceOnProviderTx(this.offering.tid, this.offering.price);
      const txHash2 = await this.metamaskService.signAndSentTransaction(unsignedAllowanceTx);

      // Wait for the allowance transaction to be confirmed
      this.isLoading = true;
      const receipt2 = await this.metamaskService.waitForTransactionReceipt(txHash2);
      this.isLoading = false;

      if (receipt2.status !== 'success') {
        console.error('Allowance transaction failed:', receipt2);
        this.transactionStatusMessage = 'Allowance transaction failed. Please try again.';
        this.snackBar.open('Allowance transaction failed. Please try again.', 'Close', { duration: 5000 });
        return;
      }
    }

    // 4. Check the receipt status and show appropriate snackbar message.
    console.log('Transaction confirmed');
    this.transactionStatusMessage = 'Transaction confirmed! Please check your Metamask Wallet';
    this.snackBar.open('Transaction confirmed', 'Close', { duration: 5000 });

  } catch (error) {
    this.isLoading = false;
    console.error('Transaction error:', error);
    this.transactionStatusMessage = 'An error occurred. Please try again.';
    this.snackBar.open('An error occurred. Please try again.', 'Close', { duration: 5000 });
  }
}

  async archiveOffering(): Promise<void> {
    const dialogRef = this.dialog.open(ArchiveOfferingDialogComponent, {
      width: '500px',
      data: {
        offeringTitle: this.offering.title,
        offeringId: this.offeringId
      }
    });

    const result = await firstValueFrom(dialogRef.afterClosed());
    if (result === true) {
      // Enter suspended state
      this.isUnpublishingLoading = true;
      this.changeDetectorRef.markForCheck();

      try {
        await lastValueFrom(this.provenanceApiService.unpublishOffering(this.offeringId));

        this.snackBar.open('Offering has been unpublished successfully', 'Success', {
          duration: 5000,
        });

        // Reload the page to show the banner (refresh offering details)
        window.location.reload();

      } catch (error) {
        // Restore previous state on error
        this.isUnpublishingLoading = false;
        this.changeDetectorRef.markForCheck();

        console.error('Error archiving offering:', error);
        this.snackBar.open('Failed to unpublish offering. Please try again.', 'Error', {
          duration: 5000,
        });
      }
    }
  }

  public priceTo2Decimals(price: any) {
    const val = Math.round(price * 100) / 100;
    return val.toFixed(2);
  }

  private decodeCap(offering: any) {
    // reset flags
    this.isSubscription = this.isPayAsYouUse = this.isPayAsYouGo = false;

    if (offering.unit.duration) {
      this.isSubscription = true;
      this.unit = offering.unit.duration;
      this.unitType = 'access duration';
      switch (offering.unit.durationType) {
        case 'm':
          this.unitAdditional = 'minutes';
          break;
        case 'H':
          this.unitAdditional = 'hours';
          break;
        case 'D':
          this.unitAdditional = 'days';
          break;
        case 'M':
          this.unitAdditional = 'months';
          break;
        case 'Y':
          this.unitAdditional = 'years';
          break;
      }
    } else if (offering.unit.retrievals) {
      this.isPayAsYouUse = true;
      this.unit = offering.unit.retrievals;
      this.unitType = 'access count';
    } else if (offering.unit.volume) {
      this.isPayAsYouGo = true;
      this.unit = offering.unit.volume;
      this.unitType = 'content volume';
      this.unitAdditional = offering.unit.volumeType;
    }
  }

  private getUnitDescription(): string {
    const formatUnit = (value: string): string =>
      value?.replace(/^(\d+)([A-Za-z]+)$/, '$1 $2');
  
    if (this.isSubscription) {
      return `${formatUnit(this.unit)} ${this.unitAdditional}`; // e.g., "1 M months"
    }
  
    if (this.isPayAsYouUse) {
      return `${this.unit}`; // e.g., "10"
    }
  
    if (this.isPayAsYouGo) {
      return formatUnit(this.unit); // e.g., "10MB" → "10 MB"
    }
    return '';
  }

  private createCorrelationId(parts: number) {
    const stringArr = [];
    for (let i = 0; i < parts; i++) {
      // tslint:disable-next-line:no-bitwise
      const S4 = (((1 + Math.random()) * 0x10000) | 0)
        .toString(16)
        .substring(1);
      stringArr.push(S4);
    }
    return stringArr.join('-');
  }

  copyTxHash(event: any) {
    event.preventDefault();
    this.clipboardManager.copy(this.txHash);
    this.snackBar.open('Data copied.', 'Info', {
      duration: this.defaultDuration,
    });
  }

  checkStatus(event: any) {
    console.log('Event', event);
    event.preventDefault();
    this.governanceApiService
    .checkRequestStatus(this.offering)
    .pipe(take(1))
    .subscribe(
        (response: any) => {
        if (!response) return;
        console.log('Response', response);

        const message =
            response.message +
            ' was created at ' +
            response.finalized +
            ' with status ' +
            response.status;
        this.snackBar.open(message, 'Request status', {
            duration: 6000,
        });
        },
        (error) => {
        console.log('Error', error);
        this.snackBar.open(JSON.stringify(error), 'Error', {
            duration: this.defaultDuration,
        });
        }
    );
  }

  get breadcrumbs(): {url: string; label: string }[] {
    return [
      { url: '/fdac', label: 'Home' },
      { url: this.offering?.asset ? '/asset/' + this.offering.asset : '', label: 'Asset details' },
      { url: '/offering-details', label: 'Offering details'},
    ]
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }
}