import { Component, OnDestroy, OnInit, ViewChild } from '@angular/core';
import { FormControl, FormGroup, NgForm, Validators } from '@angular/forms';
import { MatSnackBar } from '@angular/material/snack-bar';
import { ProvenanceApiService } from '../../services/api/provenance.api.service';
import { PatApiService } from 'src/app/services/api/pat.api.service';
import { ActivatedRoute, Router } from '@angular/router';
import { Buffer } from 'buffer';
const BASE58 = '123456789ABCDEFGHJKLMNPQRSTUVWXYZabcdefghijkmnopqrstuvwxyz';
const basex = require('base-x');
const bs58 = basex(BASE58);

@Component({
    selector: 'pat',
    templateUrl: './pat.component.html',
    styleUrls: ['./pat.component.scss'],
})
export class PatComponent implements OnInit, OnDestroy {
    private accounts: any = null;
    private defaultDuration: number = 3000;
    protected txHash: any;

    questionnaire: any;
    assetId: any;
    assetTitle: any;
    offeringId: any;
    patForm: any;
    offering: any;
    correlationId: any;
    subscribePat: any;
    subscribePat2: any;
    
    // Complete offering form state from offering definition
    offeringFormState?: {
        selectedUnit?: string;
        duration?: number;
        durationType?: string;
        retrievals?: number;
        volume?: number;
        volumeType?: string;
        title?: string;
        summary?: string;
        price?: string;
        scope?: string;
        license?: string;
        sla?: string;
        tid?: string;
        cdsinst?: string;
        correlationId?: string;
    };

    constructor(
        private snackBar: MatSnackBar,
        private patApiService: PatApiService,
        private route: ActivatedRoute,
        private provenanceApiService: ProvenanceApiService,
        private router: Router
    ) { }

    ngOnInit() {
        // Initialize with an empty FormGroup to prevent form loading error
        this.patForm = new FormGroup({});

        this.route.queryParams.subscribe((params) => {
            console.log('Params', params);
            if (!params['aid']) this.router.navigate(['/offering-definition']);
            this.offeringId = params['oid'];
            this.assetTitle = params['title'];
            this.assetId = params['aid'];
            
            // Extract complete offering form state
            this.offeringFormState = {
                selectedUnit: params['selectedUnit'],
                duration: params['duration'] ? Number(params['duration']) : undefined,
                durationType: params['durationType'],
                retrievals: params['retrievals'] ? Number(params['retrievals']) : undefined,
                volume: params['volume'] ? Number(params['volume']) : undefined,
                volumeType: params['volumeType'],
                title: params['offeringTitle'], // Offering title (user input) - distinct from asset title
                summary: params['summary'],
                price: params['formPrice'] || params['price'], // Use formPrice if available, fallback to price
                scope: params['scope'],
                license: params['license'],
                sla: params['sla'],
                tid: params['tid'],
                cdsinst: params['cdsinst'],
                correlationId: params['correlationId']
            };
            
            console.log('Offering form state:', this.offeringFormState);
        });

        if (this.offeringId) {
            this.subscribePat = this.provenanceApiService
                .getOfferingDetails(this.offeringId)
                .subscribe(
                    (response) => {
                        console.log('Offering details', response);
                        this.offering = response.offering;
                    },
                    (error) => {
                        console.log('Error Offering confirmed', error);
                    }
                );
        }
        this.callQuestions();
    }

    ngOnDestroy(): void {
        if (this.subscribePat) this.subscribePat.unsubscribe();
        if (this.subscribePat2) this.subscribePat2.unsubscribe();
    }

    private buildOfferingDefinitionQueryParams(additionalParams: any = {}): any {
        const queryParams: any = {
            aid: this.assetId,
            title: this.assetTitle,
            ...additionalParams
        };

        // Add complete form state if available
        if (this.offeringFormState) {
            const formState = this.offeringFormState;
            
            // Unit type parameters
            if (formState.selectedUnit) queryParams.selectedUnit = formState.selectedUnit;
            if (formState.duration !== undefined) queryParams.duration = formState.duration;
            if (formState.durationType) queryParams.durationType = formState.durationType;
            if (formState.retrievals !== undefined) queryParams.retrievals = formState.retrievals;
            if (formState.volume !== undefined) queryParams.volume = formState.volume;
            if (formState.volumeType) queryParams.volumeType = formState.volumeType;
            
            // All other form fields
            if (formState.title) queryParams.offeringTitle = formState.title;
            if (formState.summary) queryParams.summary = formState.summary;
            if (formState.price) queryParams.formPrice = formState.price; // Use formPrice to avoid conflict with PAT price
            if (formState.scope) queryParams.scope = formState.scope;
            if (formState.license) queryParams.license = formState.license;
            if (formState.sla) queryParams.sla = formState.sla;
            if (formState.tid) queryParams.tid = formState.tid;
            if (formState.cdsinst) queryParams.cdsinst = formState.cdsinst;
            
            // Include correlation ID if available (for PAT notification)
            if (this.correlationId) queryParams.correlationId = this.correlationId;
        }

        return queryParams;
    }

    onDiscard() {
        this.patForm.reset();
        this.router.navigate(['/offering-definition'], {
            queryParams: this.buildOfferingDefinitionQueryParams(),
        });
    }

    onSubmit() {
        if (this.patForm.valid) {
            this.doSubmit();
        } else {
            this.snackBar.open('Please fill all the required fields', 'Info', {
                duration: this.defaultDuration,
            });
        }
    }

    private createQuestionnaireForm() {
        if (this.questionnaire && this.questionnaire.questions) {
            const controls: any = {};
            for (const question of this.questionnaire.questions) {
                if (question.required)
                    controls[question.id] = new FormControl('', Validators.required);
                else controls[question.id] = new FormControl('');
            }
            this.patForm = new FormGroup(controls);
        }
    }

    private async createCorrelationId(parts: number) {
        const stringArr = [];
        for (let i = 0; i < parts; i++) {
            // tslint:disable-next-line:no-bitwise
            const S4 = (((1 + Math.random()) * 0x10000) | 0)
                .toString(16)
                .substring(1);
            stringArr.push(S4);
        }
        return stringArr.join('-');
    }

    convertJsonToBase58(jsonData: any): String {
        const jsonStr = JSON.stringify(jsonData);
        return bs58.encode(Buffer.from(jsonStr, 'utf8'));
    }

    private async callQuestions() {
        this.correlationId = await this.createCorrelationId(4);
        console.log('Correlation ID', this.correlationId);
        
        // Build request object based on selected unit type
        const request: any = {
            id: this.correlationId,
            asset: this.assetId,
        };

        // Add the appropriate property based on selected unit type
        const formState = this.offeringFormState;
        if (formState?.selectedUnit === 'Access duration' && formState.duration && formState.durationType) {
            request.duration = `${formState.duration}${formState.durationType}`;
        } else if (formState?.selectedUnit === 'Number of accesses' && formState.retrievals) {
            request.retrievals = formState.retrievals;
        } else if (formState?.selectedUnit === 'Volume of content' && formState.volume && formState.volumeType) {
            request.volume = `${formState.volume}${formState.volumeType}`;
        } else {
            // Fatal error if we cannot determine the unit type or required parameters are missing
            const errorMessage = `Fatal error: Cannot determine unit type parameters. Form state: ${JSON.stringify(formState)}`;
            console.error(errorMessage);
            this.snackBar.open('Missing unit type configuration. Please go back and complete the offering form.', 'Error', {
                duration: 5000,
            });
            this.router.navigate(['/offering-definition'], {
                queryParams: this.buildOfferingDefinitionQueryParams()
            });
            return;
        }

        console.log('callQuestions payload', JSON.stringify(request));
        const encodedString = this.convertJsonToBase58(request);
        console.log('callQuestions encoded payload', encodedString);
        this.subscribePat2 = this.patApiService
            .retrieveQuestions(encodedString)
            .subscribe(
                (response) => {
                    console.log('Response callQuestions', JSON.stringify(response));
                    if (!response) return;
                    // @ts-ignore
                    this.questionnaire = response;
                    this.createQuestionnaireForm();
                },
                (error) => {
                    this.snackBar.open(JSON.stringify(error), 'Error', {
                        duration: this.defaultDuration,
                    });
                }
            );
    }

    private doSubmit() {
        const questionnaireCompiled: any = {};
        questionnaireCompiled.cid = this.questionnaire.cid;
        questionnaireCompiled.answers = [];
        for (const question of this.questionnaire.questions) {
            const value = this.patForm.get(question.id + '').value;
            if (value === null || value === undefined || value === '') continue;
            const answer = {
                id: question.id,
                value: value,
            };
            questionnaireCompiled.answers.push(answer);
        }
        this.calculatePrice(questionnaireCompiled);
    }

    private calculatePrice(questionnaireCompiled: any) {
        questionnaireCompiled.cid = this.correlationId;
        console.log('calculatePrice payload', JSON.stringify(questionnaireCompiled));
        const encodedString = this.convertJsonToBase58(questionnaireCompiled);
        console.log('calculatePrice encoded payload', encodedString);
        
        this.patApiService.calculatePrice(encodedString).subscribe(
            (response) => {
                let num = 0;
                // @ts-ignore
                if (response && response.price) {
                    // @ts-ignore
                    num = +response.price;
                }
                this.patForm.reset();
                this.router.navigate(['/offering-definition'], {
                    queryParams: this.buildOfferingDefinitionQueryParams({ price: num }),
                });
            },
            (error) => {
                console.log('Error', error);
                this.snackBar.open(JSON.stringify(error), 'Error', {
                    duration: this.defaultDuration,
                });
                this.router.navigate(['/offering-definition'], {
                    queryParams: this.buildOfferingDefinitionQueryParams({ price: 0 }),
                });
            }
        );
    }

    back() {
        this.router.navigate(['/offering-definition'], {
            queryParams: this.buildOfferingDefinitionQueryParams(),
        });
    }

    get breadcrumbs(): { url: () => void; label: string }[] {
        return [
            { url: () => this.router.navigate(['/fdac']), label: 'Assets' },
            { url: () => this.router.navigate(['/asset', this.assetId]), label: 'Asset details' },
            { url: () => this.router.navigate(['/offering-definition'], { queryParams: this.buildOfferingDefinitionQueryParams() }), label: 'Offering definition' },
            { url: () => this.router.navigate(['/pat']), label: 'Pricing Advisory Tool' }
        ];
    }
}