import { Component, OnDestroy, OnInit } from '@angular/core';
import { MetamaskChainService } from 'src/app/services/chain/metamask.chain.service';
import { TmApiService } from '../../services/api/tm.api.service';
import {
  combineLatest,
  forkJoin,
  map,
  mergeMap,
  Subscription,
  take,
} from 'rxjs';
import { GovernanceApiService } from 'src/app/services/api/governance.api.service';
import { Address } from 'viem';

interface AccountUI {
  title: string;
  balance: number;
  id: Address;
  open: boolean;
  authorizationParty: string;
  dateOfOnboarding: string;
  isMemamaskConnected: boolean;
  allowance?: number;
  hasAllowance?: boolean;
}

@Component({
  selector: 'app-profile-page-account',
  templateUrl: './profile-page-account.component.html',
  styleUrls: ['./profile-page-account.component.scss'],
})
export class ProfilePageAccountComponent implements OnInit, OnDestroy {
  // Todo: Replace with UID from the current session - now it's hardcoded admin account ???

  constructor(
    private chainService: MetamaskChainService,
    private tmApiService: TmApiService,
    private governanceApiService: GovernanceApiService
  ) {}

  accounts: AccountUI[] = [];
  sub = new Subscription();
  currentMetaMaskAccount: Address | null = null;

  ngOnDestroy(): void {
    this.sub.unsubscribe();
  }

  ngOnInit() {
    // Get current MetaMask account
    this.sub.add(
      this.chainService.account$.subscribe((account) => {
        this.currentMetaMaskAccount = account;
      })
    );

    this.sub.add(
      combineLatest([
        this.chainService.accountList$,
        this.governanceApiService.getEnrolledAccounts(),
      ])
        .pipe(
          mergeMap(([metamaskAccountList, enroledAccounts]) => {
            return forkJoin(
              enroledAccounts.map((enrolledAccount, dataIndex: number) =>
                forkJoin({
                  balance: this.tmApiService.getPaymentTokenAmount(enrolledAccount.tid).pipe(take(1)),
                  allowance: this.tmApiService.checkApprovalAmount(enrolledAccount.tid).pipe(take(1))
                }).pipe(
                  map((res) => ({
                    title: `Account #${dataIndex + 1}`,
                    balance: res.balance.balance,
                    id: enrolledAccount.tid,
                    open: dataIndex === 0,
                    authorizationParty: 'Single-party authorization',
                    dateOfOnboarding: '10/01/2024',
                    isMemamaskConnected: metamaskAccountList.some(
                      (metamaskAccount) => {
                        return (
                          metamaskAccount.toLowerCase() ===
                          enrolledAccount.tid.toLowerCase()
                        );
                      }
                    ),
                    allowance: res.allowance,
                    hasAllowance: res.allowance > 0
                  }))
                )
              )
            ).pipe();
          })
        )
        .subscribe((accounts) => {
          this.accounts = accounts;
        })
    );
  }

  toggleAccount(index: number) {
    this.accounts[index].open = !this.accounts[index].open;
  }

  async approveAllowanceForAccount(account: AccountUI) {
    try {
      const approvalTx = await this.tmApiService.getApprovalTx();
      await this.chainService.signAndSentTransaction(approvalTx);

      // Refresh allowance for this account
      this.tmApiService.checkApprovalAmount(account.id).pipe(take(1)).subscribe((allowance: number) => {
        const accountIndex = this.accounts.findIndex(a => a.id === account.id);
        if (accountIndex !== -1) {
          this.accounts[accountIndex].allowance = allowance;
          this.accounts[accountIndex].hasAllowance = allowance > 0;
        }
      });
    } catch (error) {
      console.error('Error approving allowance:', error);
    }
  }

  isCurrentAccount(accountId: Address): boolean {
    return this.currentMetaMaskAccount?.toLowerCase() === accountId.toLowerCase();
  }
}