import { DOCUMENT, ViewportScroller } from '@angular/common';
import { Component, OnDestroy, OnInit, inject } from '@angular/core';
import { Observable, fromEvent, map } from 'rxjs';
import { AuthService } from 'src/app/auth/auth.service';
import { Address } from 'viem';
import { combineLatest, forkJoin, mergeMap, Subscription, take } from 'rxjs';
import { GovernanceApiService } from 'src/app/services/api/governance.api.service';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { MetamaskChainService } from 'src/app/services/chain/metamask.chain.service';
import { ActiveMemberRespDto } from 'src/app/services/api/dtos/active-member.resp.dto';


interface AccountUI {
  title: string;
  balance: number;
  id: Address;
  open: boolean;
  authorizationParty: string;
  dateOfOnboarding: string;
  isMemamaskConnected: boolean;
  allowance?: number;
  hasAllowance?: boolean;
}

@Component({
  selector: 'app-profile-page-overview',
  templateUrl: './profile-page-overview.component.html',
  styleUrls: ['./profile-page-overview.component.scss']
})
export class ProfilePageOverviewComponent implements OnInit, OnDestroy {

  userDetails: { uid?: string; role?: string; affiliation?: string; region?: string } | null = null;
  
  nickName: string = '';  // To store the nickname from the JWT
  region: string = '';    // To store the region from the JWT

  loadUserDetails() {
    const token = this.authService.getToken();
    if (!token) {
      // no token => user not logged in => skip
      return;
    }
    const { nickname, region } = this.authService.getNickNameAndRegion();
    if (nickname && region) {
      this.nickName = nickname;
      this.region = region;
    }
  }

  overviewItems = [
    {
      title: 'Active Accounts',
      value: 1,
      description: '',
      isPositive: true,
      icon: '../../../assets/icons/ta_accounts.png'
    },
    {
      title: 'Indexed assets',
      value: 0,
      description: '',
      isPositive: false,
      icon: '../../../assets/icons/ti_assets.png'
    },
    {
      title: 'Traded assets',
      value: 0,
      description: '',
      isPositive: true,
      icon: '../../../assets/icons/tt_assets.png'
    },
    // {
    //   title: 'Total review',
    //   value: 10,
    //   description: '',
    //   icon: '../../../assets/icons/tr.png'
    // },
    {
      title: 'Open tickets',
      value: 0,
      description: '',
      icon: '../../../assets/icons/to_tickets.png'
    },
    {
      title: 'Closed tickets',
      value: 0,
      description: '',
      icon: '../../../assets/icons/tc_tickets.png'
    }
  ];

  // Todo: Replace with UID from the current session - now it's hardcoded admin account ???

  constructor(private authService: AuthService, private chainService: MetamaskChainService,
    private tmApiService: TmApiService,
    private governanceApiService: GovernanceApiService) { }

  accounts: AccountUI[] = [];
  sub = new Subscription();
  currentMetaMaskAccount: Address | null = null;

  activeMember?: ActiveMemberRespDto;

  ngOnDestroy(): void {
    this.sub.unsubscribe();
  }

  ngOnInit() {
    this.userDetails = this.authService.getUserDetails();
    console.log(this.userDetails);
    const pid = this.userDetails!.affiliation;

    // Mock userDetails for preview purposes
    // this.userDetails = {
    //   affiliation: 'mock-affiliation-id'
    // };

    // // Previewing the logic without real API calls
    // this.accounts = [
    //   {
    //     title: 'Account #1',
    //     balance: 1234.56,
    //     id: '0x123abc',
    //     open: true,
    //     authorizationParty: 'Single-party authorization',
    //     dateOfOnboarding: '10/01/2024',
    //     isMemamaskConnected: true
    //   },
    //   {
    //     title: 'Account #2',
    //     balance: 9876.54,
    //     id: '0x456def',
    //     open: false,
    //     authorizationParty: 'Multi-party authorization',
    //     dateOfOnboarding: '15/01/2024',
    //     isMemamaskConnected: false
    //   }
    // ];

    // 1) Fetch the active‐member record
    this.sub.add(
      this.governanceApiService.getActiveMember(pid!)
        .subscribe(member => {
          this.activeMember = member;
          console.log('Active member:', member);
        }, err => {
          console.error('Could not load active member', err);
        })
    );
    this.loadUserDetails();

    // Get current MetaMask account and reload page when it changes
    this.sub.add(
      this.chainService.account$.subscribe((account) => {
        const previousAccount = this.currentMetaMaskAccount;
        this.currentMetaMaskAccount = account;

        // If account changed (and wasn't the initial load), reload the page
        if (previousAccount && previousAccount !== account) {
          window.location.reload();
        }
      })
    );

    this.sub.add(
      combineLatest([
        this.chainService.accountList$,
        this.governanceApiService.getEnrolledAccounts(),
      ])
        .pipe(
          mergeMap(([metamaskAccountList, enroledAccounts]) => {
            return forkJoin(
              enroledAccounts.map((enrolledAccount, dataIndex: number) =>
                forkJoin({
                  balance: this.tmApiService.getPaymentTokenAmount(enrolledAccount.tid).pipe(take(1)),
                  allowance: this.tmApiService.checkApprovalAmount(enrolledAccount.tid).pipe(take(1))
                }).pipe(
                  map((res) => ({
                    title: `Account #${dataIndex + 1}`,
                    balance: res.balance.balance,
                    id: enrolledAccount.tid,
                    open: dataIndex === 0,
                    authorizationParty: 'Single-party authorization',
                    dateOfOnboarding: '10/01/2024',
                    isMemamaskConnected: metamaskAccountList.some(
                      (metamaskAccount) => {
                        return (
                          metamaskAccount.toLowerCase() ===
                          enrolledAccount.tid.toLowerCase()
                        );
                      }
                    ),
                    allowance: res.allowance,
                    hasAllowance: res.allowance > 0
                  }))
                )
              )
            ).pipe();
          })
        )
        .subscribe((accounts) => {
          this.accounts = accounts;
        })
    );
  }

  toggleAccount(index: number) {
    this.accounts[index].open = !this.accounts[index].open;
  }

  async approveAllowanceForAccount(account: AccountUI) {
    try {
      const approvalTx = await this.tmApiService.getApprovalTx();
      await this.chainService.signAndSentTransaction(approvalTx);

      // Refresh allowance for this account
      this.tmApiService.checkApprovalAmount(account.id).pipe(take(1)).subscribe((allowance: number) => {
        const accountIndex = this.accounts.findIndex(a => a.id === account.id);
        if (accountIndex !== -1) {
          this.accounts[accountIndex].allowance = allowance;
          this.accounts[accountIndex].hasAllowance = allowance > 0;
        }
      });
    } catch (error) {
      console.error('Error approving allowance:', error);
    }
  }

  isCurrentAccount(accountId: Address): boolean {
    return this.currentMetaMaskAccount?.toLowerCase() === accountId.toLowerCase();
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }
}