import { DOCUMENT, ViewportScroller } from '@angular/common';
import { Component, OnInit, OnDestroy, inject, AfterViewInit } from '@angular/core';
import { Observable, Subscription, fromEvent, map } from 'rxjs';
import { AuthService } from 'src/app/auth/auth.service';
import { PurchasedOffersService, PurchasedOffer } from 'src/app/services/api/purchased-offers.service';
import { MetamaskChainService } from 'src/app/services/chain/metamask.chain.service';

@Component({
  selector: 'app-purchased-offers',
  templateUrl: './purchased-offers.component.html',
  styleUrls: ['./purchased-offers.component.scss']
})
export class PurchasedOffersComponent implements OnInit, OnDestroy, AfterViewInit {
  purchasedOffers: PurchasedOffer[] = [];
  loading = false;
  error: string | null = null;
  userAddress: string | null = null;
  private subscription?: Subscription;

  breadcrumbs = [
    { label: 'Profile', url: '/profile-overview' },
    { label: 'Purchased Offers', url: '/' },
  ];

  constructor(
    private purchasedOffersService: PurchasedOffersService,
    private authService: AuthService,
    private metamaskService: MetamaskChainService
  ) {}

  ngOnInit(): void {
    this.loadPurchasedOffers();
  }

  ngAfterViewInit(): void {
    const script = document.createElement('script');
    script.src = 'https://website-widgets.pages.dev/dist/sienna.min.js';
    script.async = true;  // or false, depending on your needs
    script.onload = () => {
      console.log('Sienna script loaded!');
    };
    document.head.appendChild(script);
  }

  ngOnDestroy(): void {
    if (this.subscription) {
      this.subscription.unsubscribe();
    }
  }

  userDetails: { uid?: string; role?: string; affiliation?: string; region?: string } | null = null;
  
  nickName: string = '';  // To store the nickname from the JWT
  region: string = '';    // To store the region from the JWT

  loadUserDetails() {
    const token = this.authService.getToken();
    if (!token) {
      // no token => user not logged in => skip
      return;
    }
    const { nickname, region } = this.authService.getNickNameAndRegion();
    if (nickname && region) {
      this.nickName = nickname;
      this.region = region;
    }
  }

  private async loadPurchasedOffers(): Promise<void> {
    const userAddress = await this.metamaskService.getAccount();
    if (!userAddress) {
      this.error = 'User address not found. Please connect your wallet.';
      return;
    }
    this.loading = true;
    this.error = null;
    this.subscription = this.purchasedOffersService
      .getPurchasedOffersWithDetails(userAddress)
      .subscribe({
        next: (offers) => {
          this.purchasedOffers = offers;
          this.loading = false;
        },
        error: (error) => {
          console.error('Error loading purchased offers:', error);
          this.error = 'Failed to load purchased offers. Please try again.';
          this.loading = false;
        },
      });
  }

  retryLoad(): void {
    this.loadPurchasedOffers();
  }

  refreshOffers(): void {
    if (!this.userAddress) return;
    
    this.loading = true;
    this.error = null;

    this.subscription = this.purchasedOffersService.refreshUserOffers(this.userAddress)
      .subscribe({
        next: (offers: PurchasedOffer[]) => {
          this.purchasedOffers = offers;
          this.loading = false;
        },
        error: (error: any) => {
          console.error('Error refreshing purchased offers:', error);
          this.error = 'Failed to refresh purchased offers. Please try again.';
          this.loading = false;
        }
      });
  }

  getOffersByType(type: 'SUB' | 'PAYG' | 'PAYU'): PurchasedOffer[] {
    return this.purchasedOffers.filter(offer => offer.type === type);
  }

  getTypeDisplayName(type: 'SUB' | 'PAYG' | 'PAYU'): string {
    const typeNames = {
      SUB: 'Subscription',
      PAYG: 'Pay As You Go',
      PAYU: 'Pay As You Use'
    };
    return typeNames[type] || type;
  }

  formatPrice(price: string | undefined): string {
    if (!price) return 'N/A';
    return `$${price}`;
  }

  formatDate(dateString: string | undefined): string {
    if (!dateString) return 'N/A';
    return new Date(dateString).toLocaleDateString();
  }

  
  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }
}