import { Component, HostListener, OnDestroy, OnInit } from '@angular/core';
import { ActivatedRoute, Router } from '@angular/router';
import { CompositeSearchService } from 'src/app/services/composite-search.service';
import { AssetWithOfferings } from 'src/app/services/composite-search.service';
import { SearchResult } from 'src/app/models/search-result.model';
import { SearchQueries, AssetsInteraction } from 'src/app/interfaces/generated-data-search.interface';


@Component({
  selector: 'app-search-page',
  templateUrl: './search-page.component.html',
  styleUrls: ['./search-page.component.scss']
})
export class SearchPageComponent implements OnInit, OnDestroy {
  query = '';
  priceRange: number[] = [10, 10000];

  fdacResponse: AssetWithOfferings[] = [];
  kaggleResponse: SearchResult[] = [];


  businessModels = [
    { name: 'SUB', label: 'Subscription' },
    { name: 'PAYG', label: 'Pay-As-You-Go' },
    { name: 'PAYU', label: 'Pay-As-You-Use' },
    { name: 'FREE', label: 'Free of Charge' }
  ];
  selectedBusinessModel: { name: string; label: string } | null = null;

  selectedOutput: any = null;
  outputFilters = [
    { name: 'Subscription' },
    { name: 'Pay-As-You-Go' },
    { name: 'Pay-As-You-Use' }
  ];

  publisher = '';

  selectedAssetTypes: string[] = [];
  assetTypes = [
    { name: 'All' },
    { name: 'Dataset' },
    { name: 'Model' },
    { name: 'Documentation' },
    { name: 'Application' },
    { name: 'Web - API' },
    { name: 'Web-based tool' },
    { name: 'Client' },
    { name: 'Library' }
  ];

  showFilters = false;
  responseFdac: any[] | null = null;
  responseKaggle: any[] | null = null;
  loading = false;
  freeAssetsOnly = false;
  isMobile = window.innerWidth <= 768;

  infoSearch: any = { query: '', filters: '' };
  filtersSearch: any = {};
  triggeredByClick = false;

  constructor(
    private compositeSearchService: CompositeSearchService,
    private route: ActivatedRoute,
    private router: Router
  ) {}

  ngOnInit() {
    this.route.queryParams.subscribe(params => {
      this.query = params['query'] || '';
      if (this.query && !this.triggeredByClick) {
        this.search();
      } else if (!this.query) {
        this.clearSearch();
      }
    });
  }

  ngOnDestroy() {
    this.router.navigate([], {
      queryParams: { query: null },
      queryParamsHandling: 'merge'
    });
    this.query = '';
    this.responseFdac = null;
    this.responseKaggle = null;
  }

  toggleFilters() {
    this.showFilters = !this.showFilters;
  }

  search(triggeredByClick = false) {
    this.triggeredByClick = triggeredByClick;
    this.loading = true;

    const basicFilters: { type: string; values: string[] }[] = [];

    if (this.selectedAssetTypes.length > 0) {
      basicFilters.push({ type: 'tag', values: this.selectedAssetTypes });
      this.filtersSearch.type = [...this.selectedAssetTypes];
    } else {
      delete this.filtersSearch.type;
    }

    if (this.publisher) {
      basicFilters.push({ type: 'developer', values: [this.publisher] });
      this.filtersSearch.developer = this.publisher;
    } else {
      delete this.filtersSearch.developer;
    }

    if (this.selectedBusinessModel) {
      basicFilters.push({
        type: 'business_model',
        values: [this.selectedBusinessModel.name]
      });
      this.filtersSearch.business_model = this.selectedBusinessModel.name;
    } else {
      delete this.filtersSearch.business_model;
    }

    if (this.selectedOutput?.name) {
      basicFilters.push({
        type: 'data_consumption_model',
        values: [this.selectedOutput.name]
      });
      this.filtersSearch.data_consumption_model = this.selectedOutput.name;
    } else {
      delete this.filtersSearch.data_consumption_model;
    }

    if (this.freeAssetsOnly) {
      this.priceRange = [0, 0];
      this.filtersSearch.free_assets = true;
    } else {
      delete this.filtersSearch.free_assets;
    }

    this.filtersSearch.price = this.priceRange;

    this.router.navigate([], {
      queryParams: { query: this.query },
      queryParamsHandling: 'merge'
    });

    // 🚀 FDAC Assets Search
    /*this.compositeSearchService
      .searchAssetsWithPrice(
        this.query,
        basicFilters,
        this.selectedBusinessModel?.name
      )
      .subscribe({
        next: fdacResults => {
          this.fdacResponse = fdacResults.map(item => ({
            ...item,
            hover: false,
            expanded: false
          }));

          
          
        },
        error: err => {
          console.error('Search error:', err);
          this.loading = false;
        }
      });*/
  }

  clearSearch() {
    this.query = '';
    this.responseFdac = null;
    this.responseKaggle = null;
  }

  clearFilters() {
    this.query = '';
    this.priceRange = [10, 10000];
    this.selectedBusinessModel = null;
    this.selectedOutput = null;
    this.selectedAssetTypes = [];
    this.publisher = '';
    this.freeAssetsOnly = false;
    this.responseFdac = null;
    this.responseKaggle = null;
    this.router.navigate([], {
      queryParams: { query: null },
      queryParamsHandling: 'merge'
    });
  }

  toggleFreeAssets() {
    this.freeAssetsOnly ? (this.priceRange = [0, 0]) : (this.priceRange = [10, 10000]);
  }

  toggleAssetType(typeName: string) {
    if (typeName === 'All') {
      this.selectedAssetTypes = [];
      return;
    }

    const index = this.selectedAssetTypes.indexOf(typeName);
    if (index === -1) {
      this.selectedAssetTypes.push(typeName);
    } else {
      this.selectedAssetTypes.splice(index, 1);
    }
  }

  toggleAllAssets() {
    this.selectedAssetTypes = [];
  }

  isSelected(typeName: string): boolean {
    if (typeName === 'All') {
      return this.selectedAssetTypes.length === 0;
    }
    return this.selectedAssetTypes.includes(typeName);
  }

  @HostListener('window:resize')
  onResize() {
    this.isMobile = window.innerWidth <= 768;
  }

  get searchButtonLabel() {
    return this.isMobile ? '' : 'Search';
  }
}