import { Component, ChangeDetectorRef, OnInit, OnDestroy } from '@angular/core';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { MetamaskChainService } from 'src/app/services/chain/metamask.chain.service';
import { Address, GetChainIdReturnType } from 'viem';
import { combineLatest, lastValueFrom, Subscription, take, tap } from 'rxjs';

@Component({
  selector: 'app-trade-account-info',
  templateUrl: './trade-account-info.component.html',
  styleUrls: ['./trade-account-info.component.scss'],
})
export class TradeAccountInfoComponent implements OnInit, OnDestroy {
  address: string | null = null;
  chain: GetChainIdReturnType | null = null;
  paymantTokenAmount = 0;
  nativeTokenAmount = 0;
  userAllowance = '0';

  account$ = this.metaWalletService.account$;
  isSupportedChain$ = this.metaWalletService.isSupportedChain$;

  constructor(
    private metaWalletService: MetamaskChainService,
    private tmApiService: TmApiService,
    private changeDetectorRef: ChangeDetectorRef
  ) {}

  private sub1: Subscription | null = null;

  ngOnInit(): void {
    this.metaWalletService.switchChainIfNeeded().then(() => {
      this.metaWalletService.getChain().then((chain) => {
        this.chain = chain;
      });
    });
    this.sub1 = combineLatest([this.account$, this.isSupportedChain$])
      .pipe(
        tap(([address, isSupportedChain]) => {
          if (!address) {
            return;
          }
          this.address = address;

          if (!isSupportedChain) {
            this.paymantTokenAmount = 0;
            this.nativeTokenAmount = 0;
            this.userAllowance = '0';
          }
          this.setAccountStats(address);
          this.changeDetectorRef.detectChanges();
        })
      )
      .subscribe((address) => {
        return address;
      });
  }

  ngOnDestroy() {
    this.sub1?.unsubscribe();
  }

  private async setAccountStats(address: Address) {
    return lastValueFrom(
      this.tmApiService.getPaymentTokenAmount(address).pipe(take(1))
    )
      .then((res) => {
        this.paymantTokenAmount = res.balance;
        this.changeDetectorRef.detectChanges();
      })
      .then(() => {
        return this.setAllowance();
      })
      .then(() => {
        return this.metaWalletService
          .getNativeTokenBalance()
          .then((balance) => {
            this.nativeTokenAmount = balance;
            return balance;
          });
      })
      .then(() => {
        this.changeDetectorRef.detectChanges();
      });
  }

  private async setAllowance() {
    if (!this.address) {
      console.error('No connected account');
      return;
    }
    return lastValueFrom(
      this.tmApiService.checkApprovalAmount(this.address).pipe(take(1))
    ).then((amount) => {
      this.userAllowance = this.formatLargeNumber(amount);
      this.changeDetectorRef.detectChanges();
    });
  }

  private formatLargeNumber(number: number): string {
    const BILLION = 1e9;

    // If the number is more than a billion, return "infinite amount"
    if (number > BILLION) {
      return 'Infinite amount';
    }

    // For numbers less than a billion, format them normally
    return new Intl.NumberFormat('en-US').format(number);
  }
}