import { Component, ChangeDetectorRef, OnInit, OnDestroy, inject } from '@angular/core';
import { TmApiService } from 'src/app/services/api/tm.api.service';
import { MetamaskChainService } from 'src/app/services/chain/metamask.chain.service';
import { Address, GetChainIdReturnType } from 'viem';
import {
  combineLatest,
  debounceTime,
  distinctUntilChanged,
  fromEvent,
  lastValueFrom,
  map,
  Observable,
  Subscription,
  take,
  tap,
} from 'rxjs';
import { MatSnackBar } from '@angular/material/snack-bar';
import { SnackbarService } from 'src/app/services/snackbar/snackbar.service';
import { FormBuilder, Validators } from '@angular/forms';
import { IsEthAddressValidator } from 'src/app/validators/is-eth-address.validator';
import { GovernanceApiService } from 'src/app/services/api/governance.api.service';
import { Router } from '@angular/router';
import { DOCUMENT, ViewportScroller } from '@angular/common';

@Component({
  selector: 'app-trade-account-setup',
  templateUrl: './trade-account-setup.component.html',
  styleUrls: ['./trade-account-setup.component.scss'],
})
export class TradeAccountSetupComponent implements OnInit, OnDestroy {

  breadcrumbs = [
    { label: 'Profile', url: '/profile-overview' },
    { label: 'Trading Account Enrollment', url: '/' },
  ];

  address: string | null = null;
  connectedAddressList: Address[] = [];
  chain: GetChainIdReturnType | null = null;
  paymantTokenAmount = 0;
  nativeTokenAmount = 0;
  userAllowance = '0';
  isSupportedChain = false;

  account$ = this.metaWalletService.account$;
  accountList$ = this.metaWalletService.accountList$;
  isSupportedChain$ = this.metaWalletService.isSupportedChain$;

  newAddress: string | null = null;

  constructor(
    private metaWalletService: MetamaskChainService,
    private tmApiService: TmApiService,
    private governanceApiService: GovernanceApiService,
    private changeDetectorRef: ChangeDetectorRef,
    private snackBar: MatSnackBar,
    private formBuilder: FormBuilder,
    private router: Router,
    private snackbarService: SnackbarService
  ) {}

  private sub1: Subscription | null = null;

  form = this.formBuilder.group({
    newAddress: ['', [Validators.required, IsEthAddressValidator()]],
    isTermAccepted: [false, Validators.requiredTrue],
  });

  isConnectedAddresSameAsProvidedAddress$: Observable<boolean> = combineLatest([
    this.form.controls.newAddress.valueChanges,
    this.account$,
  ]).pipe(
    debounceTime(100),
    map(([newAddress, currentConnectedAddress]) => {
      return (
        newAddress?.toLowerCase() ===
        currentConnectedAddress?.toLocaleLowerCase()
      );
    }),
    distinctUntilChanged()
  );

  isAlreadyEnrolled$: Observable<boolean> = combineLatest([
    this.form.controls.newAddress.valueChanges,
    this.governanceApiService.getEnrolledAccounts(),
  ]).pipe(
    debounceTime(100),
    map(([newAddress, enrolledAddresses]) => {
      return enrolledAddresses.some(
        (enrolledAddress) =>
          enrolledAddress.tid.toLowerCase() === newAddress?.toLowerCase()
      );
    }),
    distinctUntilChanged()
  );

  ngOnInit(): void {
    this.metaWalletService.switchChainIfNeeded().then(() => {
      this.metaWalletService.getChain().then((chain) => {
        this.chain = chain;
      });
    });
    this.sub1 = combineLatest([this.accountList$, this.isSupportedChain$])
      .pipe(
        tap(([addressList, isSupportedChain]) => {
          if (!addressList) {
            return;
          }
          this.address = addressList[0];
          this.connectedAddressList = addressList;

          if (!isSupportedChain) {
            this.isSupportedChain = false;
            this.paymantTokenAmount = 0;
            this.nativeTokenAmount = 0;
            this.userAllowance = '0';
          } else {
            this.isSupportedChain = true;
          }
          this.setAccountStats(addressList[0]);
          this.changeDetectorRef.detectChanges();
        })
      )
      .subscribe(() => {});
    // this.subscribeAndCheckIfAddressIsEnrolled();
  }

  ngOnDestroy() {
    this.sub1?.unsubscribe();
  }

  private subscribeAndCheckIfAddressIsEnrolled() {
    this.sub1?.add(
      this.isAlreadyEnrolled$.subscribe((isAlreadyEnrolled) => {
        if (isAlreadyEnrolled) {
          this.form.controls.newAddress.setErrors({
            alreadyEnrolled: true,
          });
        } else {
          const errors = this.form.controls.newAddress.errors;
          if (errors && errors['alreadyEnrolled']) {
            delete errors['alreadyEnrolled'];
            if (Object.keys(errors).length === 0) {
              this.form.controls.newAddress.setErrors(null);
            } else {
              this.form.controls.newAddress.setErrors(errors);
            }
          }
        }
      })
    );
  }

  private async setAccountStats(address: Address) {
    return lastValueFrom(
      this.tmApiService.getPaymentTokenAmount(address).pipe(take(1))
    )
      .then((res) => {
        this.paymantTokenAmount = res.balance;
        this.changeDetectorRef.detectChanges();
      })
      .then(() => {
        return this.setAllowance();
      })
      .then(() => {
        return this.metaWalletService
          .getNativeTokenBalance()
          .then((balance) => {
            this.nativeTokenAmount = balance;
            return balance;
          });
      })
      .then(() => {
        if (!this.address || !this.isSupportedChain) {
          this.form.controls.newAddress.disable();
        } else {
          this.form.controls.newAddress.enable();
        }
        this.changeDetectorRef.detectChanges();
      });
  }

  async approveTermsAndConditions() {
    if (!this.form.valid) {
      this.snackbarService.error('Please fill in all required fields.', 'Close', 5000);
      return;
    }

    try {
      await this.allowNewAddressInBlockchain();
      await this.approveOperator();
      this.snackbarService.success(
        'Your account was successfully registered for trading.',
        'Close',
        5000
      );
      this.router.navigate(['/profile-overview']);
    } catch (error) {
      console.error('Error during enrollment:', error);
      this.snackbarService.error(
        'An error occurred while registering your account. Please try again.',
        'Close',
        5000
      );
    }
  }

  private async allowNewAddressInBlockchain() {
    const newAddress: Address = this.form.value.newAddress as Address;
    if (!newAddress) {
      throw new Error('No address provided');
    }

    return lastValueFrom(
      this.governanceApiService.enrolAddress(newAddress).pipe(take(1))
    ).then((data) => {
      console.log(
        `Account ${newAddress} was successfully enroled to the blockchain`
      );
      return data;
    });
  }

  private async approveOperator() {
    return this.tmApiService.getApprovalTx().then((res) => {
      return this.metaWalletService.signAndSentTransaction(res);
    });
  }

  private async setAllowance() {
    if (!this.address) {
      console.error('No connected account');
      return;
    }
    return lastValueFrom(
      this.tmApiService.checkApprovalAmount(this.address).pipe(take(1))
    ).then((amount) => {
      this.userAllowance = this.formatLargeNumber(amount);
      this.changeDetectorRef.detectChanges();
    });
  }

  private formatLargeNumber(number: number): string {
    const BILLION = 1e9;

    // If the number is more than a billion, return "infinite amount"
    if (number > BILLION) {
      return 'Infinite amount';
    }

    // For numbers less than a billion, format them normally
    return new Intl.NumberFormat('en-US').format(number);
  }

  private readonly document = inject(DOCUMENT);
  private readonly viewport = inject(ViewportScroller);
  readonly showScroll$: Observable<boolean> = fromEvent(
    this.document,
    'scroll'
  ).pipe(
    map(() => this.viewport.getScrollPosition()?.[1] > 0)
  );

  onScrollToTop(): void {
    this.viewport.scrollToPosition([0, 0]);
  }
}