import { Injectable } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { environment } from '../../../environments/environment';
import { Observable } from 'rxjs';
import { StatsRespOfferingDto, OfferingRecord, OfferingDescriptor } from './dtos/offering.dto';
import { AssetDetails, AssetLineageTracing } from './dtos/asset.dto';

@Injectable({
  providedIn: 'root',
})
export class ProvenanceApiService {
  private assetUrl = environment.ptUrl + 'assets/';
  private offeringUrl = environment.ptUrl + 'offerings/';
  private statUrl = environment.ptUrl + 'stats';

  constructor(private httpClient: HttpClient) { }

  public submitAsset(asset: any) {
    console.log('submitAsset ' + JSON.stringify(asset));
    return this.httpClient.post(this.assetUrl, asset);
  }

  public resubmitAsset(assetId: string, asset: any) {
    console.log('resubmitAsset(' + assetId + ') ' + JSON.stringify(asset));
    return this.httpClient.put(this.assetUrl + assetId, asset);
  }

  public checkAssetForResubmission(assetId: string): Observable<{ canEdit: boolean; reason?: string }> {
    // TODO: Implement actual endpoint when available
    // This stub method should verify:
    // 1. Asset is not in "archived" state
    // 2. Current user has WRITE access to the asset
    console.log('checkAssetForResubmission (STUB) for assetId:', assetId);

    // For now, return a mock response indicating edit is allowed
    // In real implementation, this would call: this.httpClient.get(this.assetUrl + assetId + '/edit-permission')
    return new Observable(observer => {
      setTimeout(() => {
        observer.next({ canEdit: true });
        observer.complete();
      }, 100); // Simulate network delay
    });
  }

  public unpublishAsset(assetId: string): Observable<void> {
    return this.httpClient.delete<void>(
      this.assetUrl + assetId
    );
  }

  public getAssetDetails(assetId: string): Observable<AssetDetails> {
    return this.httpClient.get<AssetDetails>(
      this.assetUrl + assetId
    );
  }

  public getAssetLineage(assetId: string): Observable<AssetLineageTracing> {
    return this.httpClient.get<AssetLineageTracing>(
      this.assetUrl + assetId + "/tracing"
    );
  }

  public submitOffering(offering: any, patCorrelationId?: string): Observable<{ id: string }> {
    console.log('submitOffering ' + JSON.stringify(offering));
    let url = this.offeringUrl;
    if (patCorrelationId) {
      url += '?pat=' + encodeURIComponent(patCorrelationId);
      console.log('submitOffering with PAT correlation ID: ' + patCorrelationId);
    }
    return this.httpClient.post<{ id: string }>(url, offering);
  }

  public unpublishOffering(offeringId: string): Observable<void> {
    return this.httpClient.delete<void>(
      this.offeringUrl + offeringId
    );
  }

  public getOfferingDetails(offeringId: string): Observable<OfferingRecord> {
    return this.httpClient.get<OfferingRecord>(
      this.offeringUrl + offeringId
    );
  }

  getStats(): Observable<StatsRespOfferingDto> {
    return this.httpClient.get<StatsRespOfferingDto>(this.statUrl);
  }
}