import { Injectable } from '@angular/core';
import { Observable, forkJoin, of } from 'rxjs';
import { map, catchError, switchMap, shareReplay } from 'rxjs/operators';
import { TmApiService } from './tm.api.service';
import { ProvenanceApiService } from './provenance.api.service';
import { AssetDetails } from './dtos/asset.dto';

export interface PurchasedOffer {
  id: string;
  type: 'SUB' | 'PAYG' | 'PAYU';
  asset?: string;
  title?: string;
  price?: string;
  summary?: string;
  scope?: string;
  unit?: { duration: string };
  license?: string;
  sla?: string;
  tid?: string;
  cdsinst?: string;
  assetDetails?: AssetDetails;
}

@Injectable({
  providedIn: 'root'
})
export class PurchasedOffersService {
  private cache = new Map<string, Observable<PurchasedOffer[]>>();
  private readonly CACHE_DURATION = 5 * 60 * 1000; // 5 minutes

  constructor(
    private tmApiService: TmApiService,
    private provenanceApiService: ProvenanceApiService
  ) {}

  public getPurchasedOffersWithDetails(userAddress: any): Observable<PurchasedOffer[]> {
    const cacheKey = `purchased-offers-${userAddress}`;
    
    if (this.cache.has(cacheKey)) {
      return this.cache.get(cacheKey)!;
    }

    const request$ = this.fetchPurchasedOffersWithDetails(userAddress).pipe(
      shareReplay(1)
    );

    this.cache.set(cacheKey, request$);

    setTimeout(() => {
      this.cache.delete(cacheKey);
    }, this.CACHE_DURATION);

    return request$;
  }

  private fetchPurchasedOffersWithDetails(userAddress: string): Observable<PurchasedOffer[]> {
    return this.tmApiService.getCustomerPurchasedOffers(userAddress).pipe(
      switchMap((purchasedOffers: { SUB: string[]; PAYG: string[]; PAYU: string[] }) => {
        const allOffers: PurchasedOffer[] = [];
        
        Object.entries(purchasedOffers).forEach(([type, offerIds]: [string, string[]]) => {
          offerIds.forEach((id: string) => {
            allOffers.push({
              id,
              type: type as 'SUB' | 'PAYG' | 'PAYU'
            });
          });
        });

        if (allOffers.length === 0) {
          return of([]);
        }

        const offerDetailRequests = allOffers.map(offer =>
          this.provenanceApiService.getOfferingDetails(offer.id).pipe(
            switchMap((offerRecord: any) => {
              const offerDetails = offerRecord.offering; // Extract offering from OfferingRecord
              const updatedOffer = {
                ...offer,
                asset: offerDetails.asset,
                title: offerDetails.title,
                price: offerDetails.price,
                summary: offerDetails.summary,
                scope: offerDetails.scope,
                unit: offerDetails.unit,
                license: offerDetails.license,
                sla: offerDetails.sla,
                tid: offerDetails.tid,
                cdsinst: offerDetails.cdsinst
              };

              if (offerDetails.asset) {
                return this.provenanceApiService.getAssetDetails(offerDetails.asset).pipe(
                  map((assetDetails: any) => ({
                    ...updatedOffer,
                    assetDetails
                  })),
                  catchError((error: any) => {
                    console.error(`Error fetching asset details for ${offerDetails.asset}:`, error);
                    return of(updatedOffer);
                  })
                );
              }

              return of(updatedOffer);
            }),
            catchError((error: any) => {
              console.error(`Error fetching offer details for ${offer.id}:`, error);
              return of(offer);
            })
          )
        );

        return forkJoin(offerDetailRequests);
      }),
      catchError((error: any) => {
        console.error('Error fetching purchased offers:', error);
        return of([]);
      })
    );
  }

  public clearCache(): void {
    this.cache.clear();
  }

  public refreshUserOffers(userAddress: string): Observable<PurchasedOffer[]> {
    const cacheKey = `purchased-offers-${userAddress}`;
    this.cache.delete(cacheKey);
    return this.getPurchasedOffersWithDetails(userAddress);
  }
}