import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { environment } from '../../../environments/environment';
import { lastValueFrom, Observable } from 'rxjs';
import { SignTransactionParameters } from 'viem';

@Injectable({
  providedIn: 'root',
})
export class TmApiService {
  private tmUrl = environment.tmUrl;

  httpOptions = {
    headers: new HttpHeaders({
      'Content-Type': 'application/json',
    }),
  };

  constructor(private httpClient: HttpClient) {}

  public getApprovalTx(): Promise<SignTransactionParameters<any, any>> {
    return lastValueFrom(
      this.httpClient.get<SignTransactionParameters<any, any>>(
        this.tmUrl + 'payment/approve-operator-to-spend-payment-tokens',
        this.httpOptions
      )
    );
  }

  public getSubmissionOrderTx(
    oid: string
  ): Promise<SignTransactionParameters<any, any>> {
    return lastValueFrom(
      this.httpClient.post<SignTransactionParameters<any, any>>(
        this.tmUrl + 'submissions/order',
        {
          oid,
        },
        this.httpOptions
      )
    );
  }

  public inscreaseAllowanceOnProviderTx(
    toApproveAddress: string,
    amount: string
  ): Promise<SignTransactionParameters<any, any>> {
    return lastValueFrom(
      this.httpClient.get<SignTransactionParameters<any, any>>(
        this.tmUrl +
          `payment/get-increase-allowance-tx?amountToAllow=${amount}&toApproveAddress=${toApproveAddress}`,
        this.httpOptions
      )
    );
  }

  public checkApprovalAmountOnProvider(
    userAddress: string,
    providerAddress: string
  ): Observable<number> {
    return this.httpClient.get<number>(
      this.tmUrl +
        `payment/get-payment-allowance-on-address?userAddress=${userAddress}&allowanceCheckAddress=${providerAddress}`,
      this.httpOptions
    );
  }

  public checkApprovalAmount(address: string): Observable<number> {
    return this.httpClient.get<number>(
      this.tmUrl +
        `payment/get-payment-allowance-on-operator?userAddress=${address}`,
      this.httpOptions
    );
  }

  public getPaymentTokenAmount(
    address: string
  ): Observable<{ balance: number }> {
    return this.httpClient.get<{ balance: number }>(
      this.tmUrl + `contract-info/payment-balances/accounts/${address}`,
      this.httpOptions
    );
  }

  public getCustomerPurchasedOffers(
    address: string
  ): Observable<{ SUB: string[]; PAYG: string[]; PAYU: string[] }> {
    return this.httpClient.get<{ SUB: string[]; PAYG: string[]; PAYU: string[] }>(
      this.tmUrl + `purchases/offers?userAddress=${address}`,
      this.httpOptions
    );
  }

  public getTradesCount(): Observable<number> {
    return this.httpClient.get<number>(
      this.tmUrl + 'trading-history/trades-count',
      this.httpOptions
    );
  }

  handleError(error: any): Promise<any> {
    console.error('An error occurred', error);
    let message = '';

    if (error.error.message) {
      message = error.error.message;
    } else if (error.message) {
      message = error.message;
    } else {
      message = JSON.stringify(error);
    }
    alert('API call error: ' + message);
    return Promise.reject(message);
  }
}