// src/app/services/asset-policy.service.ts
import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders, HttpParams } from '@angular/common/http';
import { Observable, of } from 'rxjs';
import { catchError, map } from 'rxjs/operators';
import { AuthService } from 'src/app/auth/auth.service';
import { environment } from 'src/environments/environment';

export interface AssetPolicy {
    assetType: string;
    assetId: string;
    accessType: 'PUBLIC' | 'RESTRICTED' | 'PRIVATE' | string;
    rule?: string;
    }

    @Injectable({ providedIn: 'root' })
    export class AssetPolicyService {
    private readonly base = `${environment.apmURL}/public/api/v1.0`;

    constructor(private http: HttpClient, private auth: AuthService) {}

    // ------------------------
    // LOCAL (no HTTP) HELPERS
    // ------------------------

    /** Try to read the accessType from the FDAC asset payload. Returns null if the field is not present. */
    inferAccessTypeFromAsset(asset: any): 'PUBLIC' | 'RESTRICTED' | 'PRIVATE' | null {
        // Adjust these candidate paths to your real FDAC shape if needed
        const candidates = [
        asset?.accessType,
        asset?.visibility?.accessType,
        asset?.dataElement?.visibility?.accessType,
        asset?.policy?.accessType
        ].filter(Boolean);

        const val = String(candidates[0] ?? '').toUpperCase();
        return (val === 'PUBLIC' || val === 'RESTRICTED' || val === 'PRIVATE') ? (val as any) : null;
    }

    /** Returns true if we can assert the asset is PUBLIC from the FDAC payload; null if unknown. */
    isPublicFromAsset(asset: any): boolean | null {
        const t = this.inferAccessTypeFromAsset(asset);
        return t ? t === 'PUBLIC' : null;
    }

    /**
     * Synchronous decision used in list rendering:
     * - If user is authenticated -> show everything (by business rule).
     * - If NOT authenticated -> show only assets that are PUBLIC according to the FDAC payload.
     *   (If we can't determine accessType locally -> be conservative: do not show.)
     */
    shouldShowAsset(asset: any): boolean {
        const hasToken = !!this.auth.getToken?.();
        if (hasToken) return true;
        return this.isPublicFromAsset(asset) === true;
    }

    // ------------------------
    // APM (HTTP) – optional
    // ------------------------

    /** Returns APM policy; only works if a valid JWT is available. */
    getPolicy(assetId: string): Observable<AssetPolicy | null> {
        if (!assetId) return of(null);

        const token = (this.auth.getToken?.() || '').trim();
        const isJwt = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+$/.test(token);
        if (!isJwt) return of(null); // anonymous: APM will 400; skip

        const url = `${this.base}/asset-policy-editor`;
        const params = new HttpParams().set('assetId', assetId);
        const headers = new HttpHeaders({ Authorization: `Bearer ${token}` });

        return this.http.get<AssetPolicy>(url, { params, headers }).pipe(
        catchError(() => of(null))
        );
    }

  /** Convenience: PUBLIC / not PUBLIC / unknown via APM. */
    getAccessType(assetId: string): Observable<'PUBLIC' | 'RESTRICTED' | 'PRIVATE' | null> {
        return this.getPolicy(assetId).pipe(
        map(p => (p ? (p.accessType?.toUpperCase() as any) : null)),
        catchError(() => of(null))
        );
    }
}