import { Injectable, OnDestroy } from '@angular/core';
import { MatSnackBar } from '@angular/material/snack-bar';
import {
  BehaviorSubject,
  debounceTime,
  map,
  Observable,
  Subscription,
} from 'rxjs';
import { environment } from 'src/environments/environment';
import {
  Address,
  createPublicClient,
  createWalletClient,
  custom,
  defineChain,
  formatEther,
  GetChainIdReturnType,
  SignTransactionParameters,
  TransactionExecutionError,
  WalletClient,
} from 'viem';

@Injectable({
  providedIn: 'root',
})
export class MetamaskChainService implements OnDestroy {
  private accountSubject = new BehaviorSubject<Address | null>(null);
  private accountListSubject = new BehaviorSubject<Address[] | []>([]);
  private chainIdSubject = new BehaviorSubject<number | null>(null);

  private subsription = new Subscription();

  private besu = defineChain({
    id: environment.chainId,
    name: 'besu',
    nativeCurrency: {
      name: '',
      symbol: '',
      decimals: 18,
    },
    rpcUrls: {
      default: {
        http: [],
        webSocket: undefined,
      },
    },
  });

  private walletClient!: WalletClient;
  public publicClient: any;
  private chain: any = this.besu;
  private account: any;

  constructor(private snackBar: MatSnackBar) {
    this.setUpProvider();
    this.listenChainConnection();
  }

  ngOnDestroy(): void {
    this.subsription.unsubscribe();
  }

  private setUpProvider() {
    if (typeof (window as any).ethereum !== 'undefined') {
      try {
        this.publicClient = createPublicClient({
          chain: this.chain,
          transport: custom((window as any).ethereum!),
        });
      } catch (error) {
        console.log(
          'Something went wrong, plase check your wallet config',
          error
        );
      }

      //Create wallet client to interact with ledger.Sign transactions
      try {
        this.walletClient = createWalletClient({
          account: this.account,
          chain: this.chain,
          transport: custom((window as any).ethereum!),
        });
      } catch (error) {
        console.log('Could not create wallet client', error);
      }

      this.switchChainIfNeeded();
      this.initializeMetaMaskListeners();
      this.setInitialAccountAndChain();
    } else {
      this.snackBar.open(
        'No blockchain provider detected. Please install MetaMask Wallet.',
        'Error'
      );
    }
  }

  /**
   * Wait for the transaction receipt using viem's public client.
   * @param txHash The transaction hash.
   * @returns The transaction receipt.
   */
  async waitForTransactionReceipt(txHash: string): Promise<any> {
    if (!this.publicClient) {
      throw new Error('No public client available');
    }

    try {
      const receipt = await this.publicClient.waitForTransactionReceipt({ hash: txHash, timeout: 60_000 });
      return receipt;
    } catch (error) {
      console.error('Error waiting for transaction receipt:', error);
      throw error;
    }
  }

  private async setInitialAccountAndChain() {
    const accounts = await this.getAccounts();
    const chainId = await this.getChainId();
    this.accountSubject.next(accounts[0] as Address);
    this.accountListSubject.next(accounts as Address[]);
    this.chainIdSubject.next(chainId);
  }

  private initializeMetaMaskListeners() {
    const ethereum = (window as any).ethereum;

    ethereum.on('accountsChanged', (accounts: string[]) => {
      if (accounts.length > 0) {
        this.accountSubject.next(accounts[0] as Address);
        this.accountListSubject.next(accounts as Address[]);
      } else {
        this.accountSubject.next(null); // No account available
        this.accountListSubject.next([]);
      }
    });

    ethereum.on('chainChanged', async (chainId: string) => {
      const newChainId = parseInt(chainId, 16);
      this.chainIdSubject.next(newChainId);
    });
  }

  public get account$(): Observable<Address | null> {
    return this.accountSubject.asObservable();
  }

  public get accountList$(): Observable<Address[]> {
    return this.accountListSubject.asObservable();
  }

  public get chainId$(): Observable<number | null> {
    return this.chainIdSubject.asObservable();
  }

  public get isSupportedChain$(): Observable<boolean> {
    return this.chainIdSubject.asObservable().pipe(
      map((chainId) => chainId === environment.chainId),
      debounceTime(200)
    );
  }

  async getNativeTokenBalance(): Promise<number> {
    const account = await this.getAccount();
    const balance = formatEther(
      await this.publicClient.getBalance({
        address: account,
      })
    );
    return Number(balance);
  }

  async getNativeTokenBalanceSingle(address: string): Promise<number> {
    const balance = formatEther(
      await this.publicClient.getBalance({
        address: address,
      })
    );
    return Number(balance);
  }

  async getChain(): Promise<GetChainIdReturnType> {
    return this.walletClient.getChainId();
  }

  async switchChainIfNeeded(): Promise<number> {
    const chainId = await (window as any).ethereum!.request({
      method: 'eth_chainId',
    });
    if (chainId == environment.chainId) {
      return chainId;
    } else {
      await this.walletClient.switchChain(this.chain);
      return 0;
    }
  }

  async getAccount(): Promise<Address> {
    const [account] = await (window as any).ethereum!.request({
      method: 'eth_requestAccounts',
    });
    return account;
  }

  async getAccounts(): Promise<Address[]> {
    return (window as any).ethereum!.request({
      method: 'eth_requestAccounts',
    });
  }

  async getChainId(): Promise<number> {
    const chainId = await (window as any).ethereum!.request({
      method: 'eth_chainId',
    });
    return parseInt(chainId, 16);
  }

  async signAndSentTransaction(
    unsignedTx: SignTransactionParameters<any, any>
  ) {
    let rawTx: SignTransactionParameters<any, any> = {
      ...unsignedTx,
      account: await this.getAccount(),
    };

    console.log(`Starting to prepare raw transaction ${JSON.stringify(rawTx)}`);

    let gas: bigint;

    try {
      gas = await this.publicClient.estimateGas({
        ...rawTx,
      });
      console.log(`Gas  for transaction is ${gas}`);
    } catch (error) {
      console.log('Error getting gas price', error);
      gas = 1000000n;
    }
    rawTx = {
      ...rawTx,
      gas: gas,
    };

    try {
      const gasPrice = await this.publicClient.getGasPrice();
      console.log(`Gas Price for transaction is ${gasPrice}`);
      rawTx = {
        ...rawTx,
        gasPrice: gasPrice,
      };
    } catch (error) {
      console.error('Error getting gas price', error);
    }

    try {
      const nonce = await this.publicClient.getTransactionCount({
        address: rawTx.account,
      });
      console.log(`Nonce for transaction is ${nonce}`);
      rawTx = {
        ...rawTx,
        nonce: nonce,
      };
    } catch (error) {
      console.error('Error getting nonce', error);
    }
    console.log('Start sending transaction to the blockchain', rawTx);
    try {
      const tx = await this.walletClient.sendTransaction(rawTx);
      console.log('Transaction was successfully sent. Tx hash: ' + tx);
      return tx;
    } catch (error: any) {
      if (error instanceof TransactionExecutionError) {
        if (error.message.includes('User rejected the request')) {
          console.log('User rejected the transaction.');
        } else {
          // Handle other transaction errors
          console.error('Transaction failed:', error);
          this.snackBar.open('Error sending transaction', 'Error');
        }
      } else {
        console.error('Unexpected error:', error);
      }
      throw error;
    }
  }

  async executeAssetTransaction(assetParams: any, account: string) {
    let walletClient;
    try {
      // @ts-ignore
      if (typeof window.ethereum !== 'undefined') {
        walletClient = createWalletClient({
          chain: this.besu,
          //@ts-ignore
          transport: custom(window.ethereum!),
        });
      } else {
        console.error('Metamask is not available.');
        return;
      }
      const [address] = await walletClient.getAddresses();
      console.log('from > ' + address);
      const result = await walletClient.writeContract({
        address: assetParams.addr,
        abi: JSON.parse(assetParams.abi),
        functionName: assetParams.functionName,
        args: [
            assetParams.aid,
            assetParams.hash,
            assetParams.predecessorAid || ""
        ],
        account: address,
      });
      console.log(result);
      return result;
    } catch (error) {
      console.error(error);
      throw new Error('Error when interacting with MetaMask contract');
    }
  }

  private listenChainConnection() {
    this.subsription.add(
      this.isSupportedChain$.subscribe((isSupported) => {
        if (!isSupported) {
          this.snackBar.open('Please switch to the correct network', 'Error', {
            duration: 5000,
          });
        }
      })
    );
  }
}