// src/app/services/composite-search.service.ts
import { Injectable } from '@angular/core';
import { Observable, forkJoin, of, from } from 'rxjs';
import { switchMap, map, catchError, filter } from 'rxjs/operators';

import { AdvancedSearchService } from './advanced-search.service';
import { OfferingsService, OfferingSummary } from './offerings.service';
import { AuthService } from 'src/app/auth/auth.service';
import { AssetPolicyService } from './asset-policy.service'; // provides shouldShowAsset(...)

import { SearchResult } from '../models/search-result.model';
import { ConsoleLogger } from '@angular/compiler-cli';
import { OpenAPIService } from './open-api.service';

type AssetResult = any;

export interface OfferingParsed {
  oid: string;
  model: string;
  title: string;
  priceNum: number;
}

export interface AssetWithOfferings {
  asset: AssetResult;              // Raw FDAC asset
  bestOffering?: OfferingParsed;   // Cheapest offering after filters
  allOfferings?: OfferingParsed[]; // All offerings that passed filters
}

export interface SearchRequest {
  "expand": boolean
  "outputFilter": string[]
  "values": SearchRequestValue[]
}

export interface SearchRequestValue {
  "term": string,
  "filters": any[]
}

@Injectable({ providedIn: 'root' })
export class CompositeSearchService {
  constructor(
    private offerings: OfferingsService,
    private auth: AuthService,
    private policy: AssetPolicyService,
    private openAPI: OpenAPIService
  ) {}

  /**
   * Main search flow:
   * 1) Query FDAC using provided term + filters
   * 2) Visibility filter:
   *    - If authenticated: show all assets
   *    - If anonymous: show only assets considered PUBLIC by local policy inference
   * 3) For each visible asset, fetch offerings, normalize, filter, and pick the cheapest
   */
  searchAssetsWithPrice(
    term: string,
    filters: Array<{ type: string; values: string[] }>,
    businessModels: string[]
  ): Observable<AssetWithOfferings[]> {
    const request: SearchRequest = {
      expand: true,
      outputFilter: ['components'],
      values: [
        {
          term: term || '*',
          filters: filters
        }
      ]
    };
  
    return from(this.openAPI.searchAssets(request)).pipe(
      map((json: any) => {
        if (!json) return [];
        if (Array.isArray(json)) return json;
        if (Array.isArray(json.results)) return json.results;
        if (Array.isArray(json.data)) return json.data;
        return [];
      }),
      map((assets: AssetResult[]) => {
        const visibleAssets = assets;
  
        return visibleAssets.map(asset => {
          // take  dataElement (here you can find the offerings)
          const de = asset?.dataElement ?? asset?.asset?.dataElement;
  
          const rawOfferings: any[] = Array.isArray(de?.offerings)
            ? de.offerings
            : [];
  
          let parsed: OfferingParsed[] = rawOfferings.map(o => ({
            oid: o.oid,
            model: o.model,
            title: o.title,
            priceNum: this.parsePriceToNumber(o.price)
          }));
  
          // filter business models if you have
          if (businessModels?.length) {
            parsed = parsed.filter(o => businessModels.includes(o.model));
          }
  
          const bestOffering = parsed.length
            ? [...parsed].sort((a, b) => a.priceNum - b.priceNum)[0]
            : undefined;
  
          const enriched: AssetWithOfferings = {
            asset,
            bestOffering,
            allOfferings: parsed
          };
  
          return enriched;
        });
      }),
      catchError(err => {
        console.error('[Composite] OpenAPI searchAssets failed', err);
        return of<AssetWithOfferings[]>([]);
      })
    );
  }

  private enrichWithOfferings(
    asset: AssetResult,
    businessModel?: string[]
  ): Observable<AssetWithOfferings> {
    const aid = this.resolveAid(asset);
    if (!aid) {
      // If we cannot resolve an ID for PT offerings, return the asset as-is
      return of({ asset, bestOffering: undefined, allOfferings: [] });
    }


    return this.offerings.getOfferingsByAsset(aid).pipe(
      map((list: OfferingSummary[]) => {
        // Normalize and filter offerings
        let parsed: OfferingParsed[] = (list || []).map(o => ({
          oid: o.oid,
          model: o.model,
          title: o.title,
          priceNum: this.parsePriceToNumber(o.price)
        }));
        //console.log(parsed)


        if (businessModel) {
          parsed = parsed.filter(o => businessModel.includes(o.model));
        }


        // Pick the cheapest offering
        const bestOffering = parsed.length
          ? [...parsed].sort((a, b) => a.priceNum - b.priceNum)[0]
          : undefined;

        return { asset, bestOffering, allOfferings: parsed };
      }),
      catchError(err => {
        console.warn(`[Composite] Offerings failed for asset aid="${aid}"`, err);
        return of({ asset, bestOffering: undefined, allOfferings: [] as OfferingParsed[] });
      })
    );
  }

  private resolveAid(asset: any): string | null {
    const candidates = [
      asset?.id,
      asset?.assetId,
      asset?.dataElement?.id,
      asset?.components?.[0]?.id
    ].filter((v: any) => typeof v === 'string' && v.trim().length > 0);

    // Prefer non-UUID short IDs (often used by PT) if present
    const shortish = candidates.find(v => !v.includes('-')) || candidates[0] || null;
    return shortish;
  }

 
  private parsePriceToNumber(value: any): number {
    if (typeof value === 'number') return isFinite(value) ? value : 0;
    const str = String(value ?? '')
      .replace(/[^\d.,-]/g, '')       // remove non-numeric symbols
      .replace(/\.(?=.*\.)/g, '')     // drop thousand dots if multiple
      .replace(',', '.');             // comma decimal → dot
    const num = parseFloat(str);
    return isFinite(num) ? num : 0;
  }
  
  private normalize(s: string): string {
    return (s ?? '')
      .toString()
      .normalize('NFD')
      .replace(/\p{Diacritic}/gu, '')
      .trim()
      .toLowerCase();
  }
  private getCandidateNames(a: any): string[] {
    const names: (string | undefined)[] = [
      a?.dataElement?.name,
      a?.name,
      a?.title,
      a?.asset?.name,
      ...(Array.isArray(a?.components) ? a.components.map((c: any) => c?.name) : [])
    ];
    return names.filter((s): s is string => typeof s === 'string' && s.trim().length > 0);
  }
}