import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders } from '@angular/common/http';
import { Observable, throwError } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { SearchQueries, AssetsInteraction } from 'src/app/interfaces/generated-data-search.interface';
import { environment } from 'src/environments/environment';

@Injectable({
  providedIn: 'root'
})
export class GeneratedDataSearchService {
  private readonly apiUrl = environment.searchUrl;
  
  constructor(private http: HttpClient) { }

  getCsrfToken(): string {
    this.http.get(`${this.apiUrl}csrf`, { withCredentials: true }).subscribe();

    const name = 'csrftoken=';
    const cookies = decodeURIComponent(document.cookie).split(';');

    for (const cookie of cookies) {
      let trimmedCookie = cookie.trim();
      if (trimmedCookie.startsWith(name)) {
        return trimmedCookie.substring(name.length);
      }
    }

    return '';
  }

  dataSearch(data: SearchQueries): Observable<SearchQueries> {
    const headers = new HttpHeaders({
      'Content-Type': 'application/json; charset=utf-8',
      'X-CSRFToken': this.getCsrfToken(),
    });

    return this.http.post<SearchQueries>(
      `${this.apiUrl}search-queries`,
      data,
      {
        headers,
        withCredentials: true
      }
    ).pipe(
      catchError(error => throwError(() => error))
    );
  }

  interactionSearch(data: AssetsInteraction): Observable<AssetsInteraction> {
    const headers = new HttpHeaders({
      'Content-Type': 'application/json; charset=utf-8',
      'X-CSRFToken': this.getCsrfToken(),
    });

    return this.http.post<AssetsInteraction>(
      `${this.apiUrl}assets-interactions`,
      data,
      {
        headers,
        withCredentials: true
      }
    ).pipe(
      catchError(error => throwError(() => error))
    );
  }
}