import { Injectable } from '@angular/core';

export interface MarkdownPreviewResult {
  success: boolean;
  html?: string;
  message?: string;
  type: 'no-text' | 'validation-error' | 'no-formatting' | 'parse-error' | 'success';
}

@Injectable({
  providedIn: 'root'
})
export class MarkdownPreviewService {

  // Same validation pattern used in the component (allows newlines \r\n)
  private readonly safeMarkdownText = /^[^\x00-\x08\x0B\x0C\x0E-\x1F\x7F<>\\]*$/u;

  constructor() { }

  /**
   * Process text for Markdown preview
   * @param text The text to preview
   * @param maxLength Maximum length allowed for the field
   * @returns MarkdownPreviewResult with success status and appropriate message/html
   */
  previewMarkdown(text: string | null | undefined, maxLength: number = 5000): MarkdownPreviewResult {
    // A) No text
    if (!text || text.trim().length === 0) {
      return {
        success: false,
        type: 'no-text',
        message: 'No text to be previewed'
      };
    }

    // B) Text that doesn't pass validation
    if (text.length > maxLength) {
      return {
        success: false,
        type: 'validation-error',
        message: `Cannot exceed ${maxLength} characters`
      };
    }

    if (!this.safeMarkdownText.test(text)) {
      return {
        success: false,
        type: 'validation-error',
        message: 'Invalid characters detected. Use basic Markdown formatting only (**bold**, *italic*, # headings, lists). Avoid links, images, code blocks, and HTML markup'
      };
    }

    // C) Text without any formatting
    if (!this.hasMarkdownFormatting(text)) {
      return {
        success: false,
        type: 'no-formatting',
        message: 'There are no formatting instructions in your text, preview is not available'
      };
    }

    // D & E) Text with Markdown - try to convert
    try {
      const html = this.markdownToHtml(text);

      // Check if conversion was successful (basic validation)
      if (!html || html.trim() === text.trim()) {
        return {
          success: false,
          type: 'parse-error',
          message: 'Cannot preview: some of the formatting instructions in your text are not valid'
        };
      }

      return {
        success: true,
        type: 'success',
        html: html
      };
    } catch (error) {
      return {
        success: false,
        type: 'parse-error',
        message: 'Cannot preview: some of the formatting instructions in your text are not valid'
      };
    }
  }

  /**
   * Check if text contains Markdown formatting
   */
  private hasMarkdownFormatting(text: string): boolean {
    if (!text) return false;

    const markdownPatterns = [
      /\*\*[^*\r\n]+\*\*/,         // **bold**
      /(?<!\*)\*[^*\r\n]+\*(?!\*)/,  // *italic* (not part of **)
      /^#{1,2}\s+.+$/gm,           // # headings (only levels 1 and 2)
      /^\s*[-*+]\s+.+$/gm,         // unordered lists (global multiline)
      /^\s*\d+\.\s+.+$/gm,         // ordered lists (global multiline)
      /\r?\n\s*\r?\n/,             // paragraph breaks (double newlines with optional \r)
      /\r?\n/,                     // single newlines (line breaks)
    ];

    return markdownPatterns.some(pattern => pattern.test(text));
  }

  /**
   * Convert Markdown to HTML (simplified version for preview)
   * This mirrors the safe conversion logic from the backend
   */
  private markdownToHtml(markdown: string): string {
    let html = markdown;

    // Convert headers (only levels 1 and 2 allowed) - order matters: ## before #
    // Preserve original spacing for lossless conversion
    html = html.replace(/^(##)(\s+)(.+)$/gm, (match, hashes, spaces, content) => {
      return `<h2 data-hashes="${hashes}" data-spaces="${spaces}">${content}</h2>`;
    });
    html = html.replace(/^(#)(\s+)(.+)$/gm, (match, hashes, spaces, content) => {
      return `<h1 data-hashes="${hashes}" data-spaces="${spaces}">${content}</h1>`;
    });

    // Convert bold text
    html = html.replace(/\*\*([^*\n]+)\*\*/g, '<strong>$1</strong>');

    // Convert italic text (make sure it's not part of bold)
    html = html.replace(/(?<!\*)\*([^*\n]+)\*(?!\*)/g, '<em>$1</em>');

    // Convert unordered lists (preserving original formatting for lossless conversion)
    html = html.replace(/^(\s*[-*+]\s+.+(?:\r?\n\s*[-*+]\s+.*)*)/gm, (match) => {
      // Split by line and process each list item
      const lines = match.split(/\r?\n/);
      const listItems = lines
        .map(line => {
          // Capture original formatting details
          const lineMatch = line.match(/^(\s*)([-*+])(\s+)(.*)$/);
          if (lineMatch) {
            const [, indent, marker, spaces, content] = lineMatch;
            return {
              content: content.trim(),
              indent: indent,
              marker: marker,
              spaces: spaces
            };
          }
          return null;
        })
        .filter(item => item !== null && item!.content.length > 0) // Remove empty items
        .map(item => `<li data-marker="${item!.marker}" data-indent="${item!.indent}" data-spaces="${item!.spaces}">${item!.content}</li>`)
        .join('');
      return `<ul>${listItems}</ul>`;
    });

    // Convert ordered lists (preserving original formatting for lossless conversion)
    html = html.replace(/^(\s*\d+\.\s+.+(?:\r?\n\s*\d+\.\s+.*)*)/gm, (match) => {
      // Split by line and process each list item
      const lines = match.split(/\r?\n/);
      const listItems = lines
        .map(line => {
          // Capture original formatting details
          const lineMatch = line.match(/^(\s*)(\d+\.)(\s+)(.*)$/);
          if (lineMatch) {
            const [, indent, marker, spaces, content] = lineMatch;
            return {
              content: content.trim(),
              indent: indent,
              marker: marker,
              spaces: spaces
            };
          }
          return null;
        })
        .filter(item => item !== null && item!.content.length > 0) // Remove empty items
        .map(item => `<li data-marker="${item!.marker}" data-indent="${item!.indent}" data-spaces="${item!.spaces}">${item!.content}</li>`)
        .join('');
      return `<ol>${listItems}</ol>`;
    });

    // Handle line breaks intelligently:
    // - Don't add <br> immediately after block elements (headings, lists) unless there are extra newlines
    // - Convert regular text newlines to <br>

    // Step 1: Handle multiple newlines after block elements (user wants extra spacing)
    // Convert patterns like: </h1>\n\n or </ul>\n\n\n to: </h1><br> or </ul><br><br>
    html = html.replace(/(<\/(?:h[1-6]|ul|ol)>)\n(\n+)/g, (match, closingTag, extraNewlines) => {
      const numExtraLines = extraNewlines.length;
      return closingTag + '<br>'.repeat(numExtraLines);
    });

    // Step 2: PRESERVE single newlines after block elements for lossless conversion
    // Keep them as-is instead of removing them - they don't affect HTML rendering
    // but allow us to reconstruct the original markdown exactly

    // Step 3: Handle double newlines for paragraph breaks (but not after block elements)
    html = html.replace(/\n\s*\n/g, '</p><p>');

    // Step 4: Convert remaining single newlines to <br> (these are in regular text)
    html = html.replace(/\n/g, '<br>');

    // Wrap in paragraphs if not already wrapped in other tags
    if (!html.match(/^<[h1-6ul ol]/)) {
      html = `<p>${html}</p>`;
    }

    // Clean up empty paragraphs and fix paragraph nesting
    html = html.replace(/<p><\/p>/g, '');
    html = html.replace(/<p>(<[h1-6ul ol][^>]*>)/g, '$1');
    html = html.replace(/(<\/[h1-6ul ol]>)<\/p>/g, '$1');

    return html;
  }

  /**
   * Convert HTML back to Markdown (reverse transformation)
   * This implements the reverse of markdownToHtml() for editing existing content
   * @param html The HTML content to convert back to Markdown
   * @returns Markdown string
   */
  htmlToMarkdown(html: string): string {
    if (!html || html.trim().length === 0) {
      return '';
    }

    let markdown = html;

    // Handle multiple paragraphs better - convert each paragraph separately
    // First, split on paragraph boundaries and process each one
    const paragraphs = markdown.split(/<\/p>\s*<p[^>]*>/);

    if (paragraphs.length > 1) {
      // Process multiple paragraphs
      markdown = paragraphs.map(p => {
        // Remove paragraph tags from first and last paragraphs
        p = p.replace(/^<p[^>]*>/, '').replace(/<\/p>$/, '');
        return this.convertHtmlToMarkdownInternal(p);
      }).join('\n\n');
    } else {
      // Single paragraph or no paragraphs - remove wrapper if exists
      markdown = markdown.replace(/^<p[^>]*>(.*)<\/p>$/s, '$1');
      markdown = this.convertHtmlToMarkdownInternal(markdown);
    }

    return markdown;
  }

  /**
   * Internal method to convert HTML elements to Markdown
   * This uses lossless conversion by reading metadata attributes from HTML
   */
  private convertHtmlToMarkdownInternal(html: string): string {
    let markdown = html;

    // Convert headers back to markdown using preserved metadata
    markdown = markdown.replace(/<h1[^>]*data-hashes="([^"]*)"[^>]*data-spaces="([^"]*)"[^>]*>(.*?)<\/h1>/g,
      (match, hashes, spaces, content) => `${hashes}${spaces}${content}`);
    markdown = markdown.replace(/<h2[^>]*data-hashes="([^"]*)"[^>]*data-spaces="([^"]*)"[^>]*>(.*?)<\/h2>/g,
      (match, hashes, spaces, content) => `${hashes}${spaces}${content}`);

    // Fallback for headers without metadata (backward compatibility)
    markdown = markdown.replace(/<h1[^>]*>(.*?)<\/h1>/g, '# $1');
    markdown = markdown.replace(/<h2[^>]*>(.*?)<\/h2>/g, '## $1');

    // Convert bold text back (handle both <strong> and <b>)
    markdown = markdown.replace(/<(?:strong|b)[^>]*>(.*?)<\/(?:strong|b)>/g, '**$1**');

    // Convert italic text back (handle both <em> and <i>)
    markdown = markdown.replace(/<(?:em|i)[^>]*>(.*?)<\/(?:em|i)>/g, '*$1*');

    // Convert unordered lists back using preserved metadata
    markdown = markdown.replace(/<ul[^>]*>(.*?)<\/ul>/gs, (match, content) => {
      const listItems = content.match(/<li[^>]*>(.*?)<\/li>/gs);
      if (listItems) {
        return listItems.map((item: string) => {
          // Extract metadata attributes
          const markerMatch = item.match(/data-marker="([^"]*)"/);
          const indentMatch = item.match(/data-indent="([^"]*)"/);
          const spacesMatch = item.match(/data-spaces="([^"]*)"/);

          const marker = markerMatch ? markerMatch[1] : '-';
          const indent = indentMatch ? indentMatch[1] : '';
          const spaces = spacesMatch ? spacesMatch[1] : ' ';

          const text = item.replace(/<li[^>]*>(.*?)<\/li>/s, '$1').trim();
          // Strip any nested HTML tags from list item content
          const cleanText = text.replace(/<[^>]*>/g, '');
          return `${indent}${marker}${spaces}${cleanText}`;
        }).join('\n');
      }
      return '';
    });

    // Convert ordered lists back using preserved metadata
    markdown = markdown.replace(/<ol[^>]*>(.*?)<\/ol>/gs, (match, content) => {
      const listItems = content.match(/<li[^>]*>(.*?)<\/li>/gs);
      if (listItems) {
        return listItems.map((item: string) => {
          // Extract metadata attributes
          const markerMatch = item.match(/data-marker="([^"]*)"/);
          const indentMatch = item.match(/data-indent="([^"]*)"/);
          const spacesMatch = item.match(/data-spaces="([^"]*)"/);

          const marker = markerMatch ? markerMatch[1] : '1.';
          const indent = indentMatch ? indentMatch[1] : '';
          const spaces = spacesMatch ? spacesMatch[1] : ' ';

          const text = item.replace(/<li[^>]*>(.*?)<\/li>/s, '$1').trim();
          // Strip any nested HTML tags from list item content
          const cleanText = text.replace(/<[^>]*>/g, '');
          return `${indent}${marker}${spaces}${cleanText}`;
        }).join('\n');
      }
      return '';
    });

    // Convert line breaks back to single newlines
    markdown = markdown.replace(/<br\s*\/?>/gi, '\n');

    // Clean up any remaining HTML tags (fallback)
    markdown = markdown.replace(/<[^>]*>/g, '');

    // Decode HTML entities
    markdown = markdown.replace(/&lt;/g, '<');
    markdown = markdown.replace(/&gt;/g, '>');
    markdown = markdown.replace(/&amp;/g, '&');
    markdown = markdown.replace(/&quot;/g, '"');
    markdown = markdown.replace(/&#39;/g, "'");
    markdown = markdown.replace(/&nbsp;/g, ' ');

    // Clean up extra whitespace (more conservative for lossless conversion)
    markdown = markdown.replace(/\n{3,}/g, '\n\n'); // Max 2 consecutive newlines
    markdown = markdown.replace(/[ \t]+\n/g, '\n'); // Remove trailing spaces
    markdown = markdown.trim();

    return markdown;
  }
}