// src/app/services/offerings.service.ts
import { Injectable } from '@angular/core';
import { HttpClient, HttpHeaders, HttpParams } from '@angular/common/http';
import { Observable, of } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { AuthService } from 'src/app/auth/auth.service';
import { environment } from 'src/environments/environment';

export interface OfferingSummary {
  aid: string;
  oid: string;
  model: string;
  title: string;
  price: string;
}

@Injectable({ providedIn: 'root' })
export class OfferingsService {
  private readonly baseUrl = environment.ptUrl;

  constructor(private http: HttpClient, private auth: AuthService) {}

  getOfferingsByAsset(assetId: string): Observable<OfferingSummary[]> {
    if (!assetId) return of([]);

    // Read token from AuthService (source of truth). Do NOT modify AuthService.
    const raw = this.auth.getToken?.() || '';
    const token = raw.trim();

    // Optional: token format sanity-check (helps catch copy/paste or storage issues)
    const isJwt = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+$/.test(token);
    if (!isJwt) {

      console.warn('[Offerings] No valid JWT from AuthService; skipping Authorization header.');
    }

    const url = `${this.baseUrl.replace(/\/+$/, '')}/offerings`;
    const params = new HttpParams().set('aid', assetId);

    const headers = isJwt
      ? new HttpHeaders({ Accept: 'application/json', Authorization: `Bearer ${token}` })
      : new HttpHeaders({ Accept: 'application/json' }); // keep GET lean; no Content-Type

    return this.http.get<OfferingSummary[]>(url, { params, headers }).pipe(
      catchError(err => {
        console.warn('[Offerings] GET /offerings failed', { status: err?.status, assetId });
        return of([]);
      })
    );
  }

  getOfferingsBulk(ids: string[]): Observable<OfferingSummary[]> {
    if (!ids?.length) return of([]);
  
    const url = `${this.baseUrl.replace(/\/+$/, '')}/offerings`;
  
    const raw = this.auth.getToken?.() || '';
    const token = raw.trim();
    const isJwt = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+$/.test(token);
  
    const headers = isJwt
      ? new HttpHeaders({
          'Accept': 'application/json',
          'Authorization': `Bearer ${token}`,
          'Content-Type': 'application/json'
        })
      : new HttpHeaders({
          'Accept': 'application/json',
          'Content-Type': 'application/json'
        });
  
    const body = { ids };
  
    return this.http.put<OfferingSummary[]>(url, body, { headers }).pipe(
      catchError(err => {
        console.warn('[Offerings] BULK PUT /offerings failed', {
          status: err?.status,
          ids
        });
        return of([]);
      })
    );
  }

  getAssetEntry(assetId: string): Observable<any> {
    if (!assetId) return of([]);

    const url = `${this.baseUrl.replace(/\/+$/, '')}/assets`;
    const params = new HttpParams().set('id', assetId);



    // Read token from AuthService (source of truth). Do NOT modify AuthService.
    const raw = this.auth.getToken?.() || '';
    const token = raw.trim();

    // Optional: token format sanity-check (helps catch copy/paste or storage issues)
    const isJwt = /^[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+\.[A-Za-z0-9-_]+$/.test(token);
    if (!isJwt) {

      console.warn('[Offerings] No valid JWT from AuthService; skipping Authorization header.');
    }

    const headers = isJwt
      ? new HttpHeaders({ Accept: 'application/json', Authorization: `Bearer ${token}` })
      : new HttpHeaders({ Accept: 'application/json' }); // keep GET lean; no Content-Type

      return this.http.get<OfferingSummary[]>(url, { params, headers }).pipe(
        catchError(err => {
          console.warn('[Offerings] GET /offerings failed', { status: err?.status, assetId });
          return of([]);
        })
      );
  }
}