import { Injectable } from "@angular/core";
import { AuthService } from "../auth/auth.service";
import { environment } from "src/environments/environment";

export interface APIResponse<T = any> {
    value: T,
    status: number
}

export interface SearchRequest {
    "expand": boolean
    "outputFilter": string[]
    "values": SearchRequestValue[]
}

export interface SearchRequestValue {
    "term": string,
    "filters": any[]
}

@Injectable({
    providedIn: 'root',
})
export class OpenAPIService {
    constructor(private authService: AuthService) {
    }
    async generateAssetListIframeURL(itemsPerPage?: number): Promise<APIResponse<string>> {
        const token = this.authService.getToken() || ""
        const headers = { Authorization: token }
        let url = `${environment.openAPIURL}/generateAssetListIframeURL?origin=${encodeURIComponent(window.location.origin)}`
        if (itemsPerPage) url = `${url}&itemsPerPage=${itemsPerPage}`
        const response = await fetch(url, { headers })
        const value = await response.text()
        return {
            value,
            status: response.status
        }
    }

    async generateTrainingResourceListIframeURL(projectIds?: string[]) {
        let url = `${environment.openAPIURL}/generateTrainingResourceListIframeURL?origin=${encodeURIComponent(window.location.origin)}`
        if (projectIds) url = `${url}&projectIds=${projectIds.join(",")}`
        const response = await fetch(url)
        const value = await response.text()
        return {
            value,
            status: response.status
        }
    }

    async generateAssetDetailsIframeURL(assetId: string): Promise<APIResponse<string>> {
        const token = this.authService.getToken() || ""
        const headers = { Authorization: token }
        const url = `${environment.openAPIURL}/generateAssetDetailsIframeURL/${assetId}?origin=${encodeURIComponent(window.location.origin)}`
        const response = await fetch(url, { headers })
        const value = await response.text()
        return {
            value,
            status: response.status
        }

    }

    async generateTrainingResourceDetailsIframeURL(trainingResourceId: string): Promise<APIResponse<string>> {

        const url = `${environment.openAPIURL}/generateTrainingResourceDetailsIframeURL/${trainingResourceId}?origin=${encodeURIComponent(window.location.origin)}`
        const response = await fetch(url)
        const value = await response.text()
        return {
            value,
            status: response.status
        }

    }

    async generateValidationDetailsIframeURL(validationId: string): Promise<APIResponse<string>> {

        const url = `${environment.openAPIURL}/generateValidationDetailsIframeURL/${validationId}?origin=${encodeURIComponent(window.location.origin)}`
        const response = await fetch(url)
        const value = await response.text()
        return {
            value,
            status: response.status
        }

    }

    async getAssetDetails(assetId: string, extraFields?: string[]): Promise<APIResponse> {
        const token = this.authService.getToken() || ""
        const headers = { Authorization: token }
        let url = `${environment.openAPIURL}/assetDetails/${assetId}`
        if (extraFields) url = `${url}?extraFields=${extraFields.join(",")}`
        const response = await fetch(url, { headers })
        const { status } = response
        let value: any
        if (status === 200) value = await response.json()
        else value = await response.text()

        return {
            status, value
        }
    }

    async getCount(collectionName: string, tagNames?: string[]): Promise<number> {
        if (!collectionName) return 0
        let url = `${environment.openAPIURL}/count?collectionName=${encodeURIComponent(collectionName)}`
        if (tagNames) url = `${url}&tagNames=${encodeURIComponent(tagNames.join(","))}`
        const response = await fetch(url)
        if (response.status !== 200) {
            const text = await response.text()
            throw Error(`Failed to obtain count: "${text}"`)
        }
        const text = await response.text()
        return Number(text)
    }

    private addWildcardToRequest(request: SearchRequest): SearchRequest {
        const { expand, outputFilter, values } = request
        const searchRequestValues: SearchRequestValue[] = []
        const wildcard = "*"
        for (const value of values) {
            let term = value.term
            if (term !== wildcard && !term.includes('"')) {
                if (!term.startsWith("*")) term = `${wildcard}${term}`
                if (!term.endsWith("*")) term = `${term}${wildcard}`
            }
            searchRequestValues.push({ term, filters: value.filters })
        }
        return {
            expand,
            outputFilter,
            values: searchRequestValues
        }

    }

    async searchAssets(request: SearchRequest) {
        const url = `${environment.openAPIURL}/searchAssets`
        const body = JSON.stringify(this.addWildcardToRequest(request))
        const token = this.authService.getToken() || ""
        const headers = { Authorization: token, "Content-Type": "application/json" }
        const response = await fetch(url, { headers, method: "POST", body })
        const json = await response.json()
        return json || null
    }
}