import { Directive } from '@angular/core';
import { AbstractControl, ValidationErrors, Validator, NG_VALIDATORS } from '@angular/forms';

// Regex patterns for validation
// Note: & character is allowed as it's commonly used in legitimate text (e.g., "R&D", "Terms & Conditions")
const SAFE_TEXT_ONE_LINE_REGEX = /^[^\x00-\x1F\x7F<>{}\\]+$/u;
const SAFE_TEXT_MULTI_LINE_REGEX = /^[^\x00-\x1F\x7F<>{}\\]*$/u;
const SAFE_MARKDOWN_TEXT_REGEX = /^[^\x00-\x08\x0B\x0C\x0E-\x1F\x7F<>\\]*$/u;

// Safe text validator for single-line text (no newlines allowed)
@Directive({
  selector: '[safeTextValidator]',
  providers: [{ provide: NG_VALIDATORS, useExisting: SafeTextValidatorDirective, multi: true }]
})
export class SafeTextValidatorDirective implements Validator {
  validate(control: AbstractControl): ValidationErrors | null {
    const value = control.value;
    if (!value) return null;

    if (!SAFE_TEXT_ONE_LINE_REGEX.test(value)) {
      return { 'safeText': true };
    }
    return null;
  }
}

// Safe text validator for multi-line text (newlines allowed)
@Directive({
  selector: '[safeTextMultiLineValidator]',
  providers: [{ provide: NG_VALIDATORS, useExisting: SafeTextMultiLineValidatorDirective, multi: true }]
})
export class SafeTextMultiLineValidatorDirective implements Validator {
  validate(control: AbstractControl): ValidationErrors | null {
    const value = control.value;
    if (!value) return null;

    if (!SAFE_TEXT_MULTI_LINE_REGEX.test(value)) {
      return { 'safeTextMultiLine': true };
    }
    return null;
  }
}

// Safe markdown text validator (allows markdown formatting)
@Directive({
  selector: '[safeMarkdownValidator]',
  providers: [{ provide: NG_VALIDATORS, useExisting: SafeMarkdownValidatorDirective, multi: true }]
})
export class SafeMarkdownValidatorDirective implements Validator {
  validate(control: AbstractControl): ValidationErrors | null {
    const value = control.value;
    if (!value) return null;

    if (!SAFE_MARKDOWN_TEXT_REGEX.test(value)) {
      return { 'safeMarkdown': true };
    }
    return null;
  }
}

// Array of all validator directives for easy importing
export const TEXT_VALIDATOR_DIRECTIVES = [
  SafeTextValidatorDirective,
  SafeTextMultiLineValidatorDirective,
  SafeMarkdownValidatorDirective
];