# app/__init__.py

from apscheduler.schedulers.background import BackgroundScheduler
from apscheduler.triggers.interval import IntervalTrigger
from flask import Flask, render_template
from flask_sqlalchemy import SQLAlchemy
from .config import config
import os
from datetime import datetime, timedelta, UTC
import logging
from logging.handlers import RotatingFileHandler

# Logger configuration
logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)
if not logger.handlers:
    # Console handler
    ch = logging.StreamHandler()
    ch.setLevel(logging.INFO)
    # Simple formatter (you can customize it)
    formatter = logging.Formatter(
        '[%(asctime)s] %(levelname)s in %(module)s: %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S'
    )
    ch.setFormatter(formatter)
    logger.addHandler(ch)

db = SQLAlchemy()

# Function to create the initial user and organization
def create_initial_user_and_organization(app):
    with app.app_context():
        from .models import User, Organization, PostalAddress
        from .services import OrganizationService
        logger.info("Verifying if the initial user already exists...")  # Log
        
        # Check if a user with the specified email already exists
        if not User.query.filter_by(email='ernesto.troiano@gft.com').first():
            logger.info("Creating initial user and organization...")  # Log
            try:
                # Create the user
                userGFT = User(
                    first_name='Ernesto',
                    last_name='Troiano',
                    email='ernesto.troiano@gft.com',
                    password='Test123!',
                    phone=None,
                    email_confirmed=True
                )
                db.session.add(userGFT)
                db.session.flush()  # Get the ID without committing
                logger.info("Initial user created and flushed to session")  # Log
                
                # Create the organization
                orgGFT = Organization(
                    legal_name='GFT Italia Srl',
                    org_type='FPO',
                    delegate_authority_pid='0000000000000000000000',
                    oa_status='FAME_ROA',
                    representative_id=userGFT.id,
                    status='approved',
                    onboarded_at=datetime.now(UTC) - timedelta(days=30),
                    tin='00819200478'
                )
                db.session.add(orgGFT)
                db.session.flush()
                logger.info("Organization created and flushed to session")  # Log
                
                # Create the address
                addressGFT = PostalAddress(
                    organization_id=orgGFT.id,
                    address_line1='Via Sile 18',
                    city='Milano',
                    post_code='20139',
                    country='IT'
                )
                db.session.add(addressGFT)
                logger.info("Postal address created and added to session")  # Log
                
                # Comment for deployment, this commit is for testing the application before calling the API
                #db.session.commit()
                
                # Register the organization with the external API
                success, error = OrganizationService.register_approved_member(orgGFT)
                if not success:
                    db.session.rollback()
                    logger.error(f"API registration failed for GFT organization: {error}")  # Log
                else:
                    db.session.commit()
                    logger.info("Initial user and organization successfully registered and committed")  # Log
            except Exception as e:
                db.session.rollback()
                logger.error(f"Exception occurred during user/org creation: {e}")  # Log
        else:
            logger.info("Initial user already exists, skipping creation")  # Log


def create_app(config_name='production'):
    app = Flask(__name__)
    
    # Load config
    app.config.from_object(config[config_name])
    logger.info(f"Configuration '{config_name}' loaded")  # Log
    
    # Ensure directories exist
    directories = [
        app.config['DATA_DIR'],
        app.config['DB_DIR'],
        app.config['UPLOAD_FOLDER'],
        app.config['LOGO_FOLDER']  # NEW: Add logos folder
    ]
    for directory in directories:
        if not os.path.exists(directory):
            os.makedirs(directory, exist_ok=True)
            logger.info(f"Created missing directory: {directory}")
    
    # Initialize extensions
    db.init_app(app)
    logger.info("SQLAlchemy initialized")  # Log
    
    # Import models BEFORE creating the database!
    from .models import user, organization, vote, document, marketplace_user
    from .models import digital_euro_account  # ← NEW: Import wallet models
    logger.info("Models imported")  # Log
    
    # Automatically create the database
    with app.app_context():
        db.create_all()
        logger.info("Database tables created")  # Log
        
        # Create the initial user and organization
        create_initial_user_and_organization(app)
    
    # Ensure upload folder exists
    if not os.path.exists(app.config['UPLOAD_FOLDER']):
        os.makedirs(app.config['UPLOAD_FOLDER'])
        logger.info(f"Created upload folder: {app.config['UPLOAD_FOLDER']}")  # Log
    
    # Register blueprints
    from .routes.main import main as main_blueprint
    app.register_blueprint(main_blueprint)
    logger.info("Main blueprint registered")  # Log
    
    from .routes.auth import auth as auth_blueprint
    app.register_blueprint(auth_blueprint)
    logger.info("Auth blueprint registered")  # Log
    
    from .routes.api import api as api_blueprint
    app.register_blueprint(api_blueprint, url_prefix='/api/v1')
    logger.info("API blueprint registered with prefix /api/v1")  # Log
    
    from .routes.internal_api import internal_api as internal_api_blueprint
    app.register_blueprint(internal_api_blueprint)
    logger.info("Internal API blueprint registered")  # Log
    
    from .routes.public_api import public_api as public_api_blueprint
    app.register_blueprint(public_api_blueprint, url_prefix='/api/public')
    logger.info("Public API blueprint registered with prefix /api/public")  # Log
    
    # NEW: Register wallet blueprint
    from .routes.wallet import bp as wallet_blueprint
    app.register_blueprint(wallet_blueprint)
    logger.info("Wallet blueprint registered")  # Log
    
    # Inizializza lo scheduler
    scheduler = BackgroundScheduler()
    logger.info("Scheduler initialized")  # Log
    
    # Define the function that checks for expired votes
    def check_expired_votes_job():
        with app.app_context():
            from .services import OrganizationService
            OrganizationService.check_expired_votes()
            logger.info("Scheduled job: check for expired votes completed")  # Log
    
    # Add the job to perform the check every X minutes
    scheduler.add_job(
        func=check_expired_votes_job,
        trigger=IntervalTrigger(minutes=10),  # Esegui ogni 10 minuti
        id='check_votes_job',
        name='Check expired votes',
        replace_existing=True
    )
    logger.info("Scheduled job 'check_votes_job' added")  # Log
    
    # Start the scheduler
    scheduler.start()
    logger.info("Scheduler started")  # Log
    
    # Register error handlers
    @app.errorhandler(404)
    def not_found_error(error):
        logger.warning(f"404 error: {error}")  # Log
        return render_template('errors/404.html'), 404
    
    @app.errorhandler(500)
    def internal_error(error):
        db.session.rollback()
        logger.error(f"500 error: {error}")  # Log
        return render_template('errors/500.html'), 500
    
    # Initialize CLI commands
    @app.cli.command("init-db")
    def init_db():
        """Initialize the database."""
        db.create_all()
        logger.info("Database initialized via CLI command")  # Log
    
    return app
