# app/models/digital_euro_account.py
"""
Digital Euro Account Models
Link between FAME organizations and Digital Euro wallets
"""

from app import db
from datetime import datetime


class DigitalEuroAccount(db.Model):
    """
    Digital Euro Account - ONE per Organization
    Used to fund multiple trading accounts within the organization
    """
    __tablename__ = 'digital_euro_accounts'
    
    id = db.Column(db.Integer, primary_key=True)
    
    # Linked to Organization (Member), not User
    organization_id = db.Column(db.Integer, db.ForeignKey('organizations.id'), nullable=False, unique=True)  # ← CORRETTO
    
    # Digital Euro Account Info (from DESP/BCE)
    holding_id = db.Column(db.String(128), unique=True, nullable=False)  # BCE UUID
    euban = db.Column(db.String(34), unique=True, nullable=False)        # EU IBAN format
    holding_type = db.Column(db.String(20), default='MERCHANT')          # MERCHANT for organizations
    
    # Who created/linked this account
    created_by_user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    
    # Status
    is_active = db.Column(db.Boolean, default=True)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    updated_at = db.Column(db.DateTime, default=datetime.utcnow, onupdate=datetime.utcnow)
    
    # Relationships
    organization = db.relationship('Organization', backref=db.backref('digital_euro_account', uselist=False))
    created_by = db.relationship('User', foreign_keys=[created_by_user_id])
    
    def __repr__(self):
        return f'<DigitalEuroAccount {self.euban} for Org {self.organization_id}>'
    
    def to_dict(self):
        return {
            'id': self.id,
            'organization_id': self.organization_id,
            'organization_name': self.organization.legal_name if self.organization else None,
            'holding_id': self.holding_id,
            'euban': self.euban,
            'holding_type': self.holding_type,
            'is_active': self.is_active,
            'created_at': self.created_at.isoformat() if self.created_at else None
        }


class WalletTransaction(db.Model):
    """Store all wallet transactions (Digital Euro + FAME)"""
    __tablename__ = 'wallet_transactions'
    
    id = db.Column(db.Integer, primary_key=True)
    
    # Organization context
    organization_id = db.Column(db.Integer, db.ForeignKey('organizations.id'), nullable=False)  # ← CORRETTO
    
    # User who initiated the transaction
    user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    
    # Trading account involved
    trading_account_id = db.Column(db.String(128), nullable=False)  # TID from GOV API
    trading_account_owner = db.Column(db.String(128), nullable=True)  # UID from GOV API
    
    # Transaction Info
    transaction_type = db.Column(db.String(20), nullable=False)  # 'purchase', 'redeem'
    amount = db.Column(db.Numeric(10, 2), nullable=False)
    currency = db.Column(db.String(10), default='EUR')
    
    # External IDs
    de_payment_id = db.Column(db.String(128), nullable=True)           # Digital Euro payment ID
    marketplace_tx_id = db.Column(db.String(128), nullable=True)       # FAME Marketplace transaction ID
    
    # Details
    description = db.Column(db.Text, nullable=True)
    status = db.Column(db.String(20), default='pending')  # pending, completed, failed
    error_message = db.Column(db.Text, nullable=True)
    
    # Timestamps
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    completed_at = db.Column(db.DateTime, nullable=True)
    
    # Relationships
    organization = db.relationship('Organization', backref=db.backref('wallet_transactions', lazy='dynamic'))
    user = db.relationship('User', backref=db.backref('initiated_transactions', lazy='dynamic'))
    
    def __repr__(self):
        return f'<WalletTransaction {self.transaction_type} {self.amount} for TID {self.trading_account_id}>'
    
    def to_dict(self):
        return {
            'id': self.id,
            'transaction_type': self.transaction_type,
            'amount': float(self.amount),
            'currency': self.currency,
            'trading_account_id': self.trading_account_id,
            'trading_account_owner': self.trading_account_owner,
            'de_payment_id': self.de_payment_id,
            'marketplace_tx_id': self.marketplace_tx_id,
            'description': self.description,
            'status': self.status,
            'error_message': self.error_message,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'completed_at': self.completed_at.isoformat() if self.completed_at else None,
            'user': {
                'id': self.user.id,
                'email': self.user.email
            } if self.user else None
        }
