# app/models/organization.py
from datetime import datetime
import re
from .. import db
from .postal_address import PostalAddress

# Constant for the FAME ROA PID
FAME_ROA_PID = "0000000000000000000000"  # UUID nil

class Organization(db.Model):
    __tablename__ = 'organizations'

    # Primary key - keep the integer ID for internal use
    id = db.Column(db.Integer, primary_key=True)
    
    # DID - Decentralized Identifier
    did = db.Column(db.String(200), nullable=True, index=True)
    
    # Delegate onboarding authority PID (if not self)
    delegate_authority_pid = db.Column(db.String(36), nullable=True)
    
    # PID - Provenance Identifier (UUID format)
    pid = db.Column(db.String(36), unique=True, nullable=True, index=True)

    # Legal name of the organization (REQUIRED, 1-160 characters)
    legal_name = db.Column(db.String(160), nullable=False)
    
    # Organization type (FPO, NPO, RES, EDU, LEO, PUB, UKN)
    org_type = db.Column(db.String(10), nullable=False)
    
    # Federation status (created, pending, approved, rejected)
    status = db.Column(db.String(20), default='created', index=True)
    
    # NEW: Organization logo filename (stored in data/logos/)
    logo = db.Column(db.String(255), nullable=True)
    
    # NEW: Organization website URL
    website_url = db.Column(db.String(500), nullable=True)
    
    # Timestamps
    created_at = db.Column(db.DateTime, default=datetime.utcnow)
    voting_ends_at = db.Column(db.DateTime, nullable=True)
    onboarded_at = db.Column(db.DateTime, nullable=True)
    offboarded_at = db.Column(db.DateTime, nullable=True)
    
    # Legal identifiers (at least one is required)
    lei = db.Column(db.String(20), nullable=True)  # Legal Entity Identifier
    duns = db.Column(db.String(9), nullable=True)  # Data Universal Numbering System
    bic = db.Column(db.String(11), nullable=True)  # Business Identifier Code
    tin = db.Column(db.String(40), nullable=True)  # Taxpayer Identification Number
    
    # Onboarding Authority status: 
    # 'OA' = Autonomous OA (has own DID)
    # 'DELEGATE' = Delegates to another OA (specify delegate_authority_pid)
    # 'FAME_ROA' = Uses FAME Root OA (nil UUID as delegate_authority_pid)
    oa_status = db.Column(db.String(10), nullable=False, default='FAME_ROA')
        
    # Representative user ID
    representative_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    
    # Relationships
    legal_address = db.relationship('PostalAddress', uselist=False, backref='organization',
                                 cascade='all, delete-orphan')
    votes = db.relationship('Vote', backref='organization', lazy='dynamic',
                          cascade='all, delete-orphan')
    documents = db.relationship('Document', backref='organization', lazy='dynamic',
                              cascade='all, delete-orphan')
    marketplace_users = db.relationship('MarketplaceUser', backref='organization', lazy='dynamic',
                                     cascade='all, delete-orphan')
    
    def __init__(self, *args, **kwargs):
        
        if 'pid' in kwargs and kwargs['pid']:
            self.pid = kwargs['pid']
        else:
            self.pid = None  # Leave the value NULL until it comes from the API
            
        # Process OA status and related fields
        oa_status = kwargs.pop('oa_status', 'FAME_ROA')
        
        # Set delegate authority based on OA status
        if oa_status == 'OA':
            # Autonomous OA: did required, no delegate_authority_pid
            kwargs['delegate_authority_pid'] = None
        elif oa_status == 'DELEGATE':
            # Delegate to another OA: delegate_authority_pid required, no DID
            if 'delegate_authority_pid' not in kwargs or not kwargs['delegate_authority_pid']:
                raise ValueError("delegate_authority_pid is required for DELEGATE oa_status")
            kwargs['did'] = None
        elif oa_status == 'FAME_ROA':
            # Use FAME ROA: set delegate_authority_pid to nil UUID, no DID
            kwargs['delegate_authority_pid'] = FAME_ROA_PID
            kwargs['did'] = None
        
        kwargs['oa_status'] = oa_status
        
        super(Organization, self).__init__(*args, **kwargs)
        
        #if 'voting_ends_at' not in kwargs and self.status == 'pending':
        #    from ..config import Config
        #    from datetime import timedelta
        #    self.voting_ends_at = datetime.utcnow() + timedelta(minutes=Config.VOTING_DURATION_MINUTES)
    
    @property
    def is_onboarding_authority(self):
        """
        Legacy compatibility property
        Returns True if this organization is an autonomous OA
        """
        return self.oa_status == 'OA'
    
    @staticmethod
    def validate_legal_name(value):
        """Validate legal name field"""
        if not value or len(value) < 1 or len(value) > 160:
            return False, "Legal name must be between 1 and 160 characters"
            
        # Match Latin characters, numbers, punctuation, and spaces (no newlines)
        # Use r"..." to create a raw string where escapes are ignored
        pattern = r'^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$'
        try:
            if not re.match(pattern, value, re.UNICODE):
                return False, "Legal name must only include Latin letters, numbers, punctuation, and spaces (no newlines)"
        except:
            # Fallback to a simpler pattern if the Unicode regex fails
            if not re.match(r'^[A-Za-z0-9\s\.,\-_\'\"()]+$', value):
                return False, "Legal name must only include letters, numbers, spaces and basic punctuation"
                
        return True, None
    
    @staticmethod
    def validate_lei(value):
        """Validate LEI format"""
        if not value:
            return True, None  # Optional field
            
        if len(value) != 20:
            return False, "LEI must be exactly 20 alphanumeric characters"
            
        if not value.isalnum():
            return False, "LEI must contain only alphanumeric characters"
            
        return True, None
    
    @staticmethod
    def validate_duns(value):
        """Validate DUNS format"""
        if not value:
            return True, None  # Optional field
            
        if len(value) != 9:
            return False, "DUNS must be exactly 9 numeric characters"
            
        if not value.isdigit():
            return False, "DUNS must contain only numeric characters"
            
        return True, None
    
    @staticmethod
    def validate_bic(value):
        """Validate BIC format"""
        if not value:
            return True, None  # Optional field
            
        if len(value) not in [8, 11]:
            return False, "BIC must be 8 or 11 alphanumeric characters"
            
        if not value.isalnum():
            return False, "BIC must contain only alphanumeric characters"
            
        return True, None
    
    @staticmethod
    def validate_tin(value):
        """Validate TIN format"""
        if not value:
            return True, None  # Optional field
            
        if len(value) < 1 or len(value) > 40:
            return False, "TIN must be between 1 and 40 characters"
            
        pattern = r'^(?=.*[A-Za-z0-9])[A-Za-z0-9 ]+$'
        if not re.match(pattern, value):
            return False, "TIN must contain only alphanumeric characters and spaces, and cannot consist solely of spaces"
            
        return True, None
    
    def has_valid_identifier(self):
        """Check if at least one identifier is present"""
        return any([self.lei, self.duns, self.bic, self.tin])

    
    def close_voting(self):
        """Close voting and determine outcome based on votes"""
        if self.status != 'pending' or datetime.utcnow() < self.voting_ends_at:
            return self.status

        from ..config import Config
        
        yes_votes = self.votes.filter_by(vote='yes').count()
        no_votes = self.votes.filter_by(vote='no').count()
        total_votes = yes_votes + no_votes

        if total_votes < Config.MINIMUM_VOTES_REQUIRED:
            self.status = 'rejected'
            return self.status

        approval_percentage = yes_votes / total_votes if total_votes > 0 else 0
        
        if approval_percentage > Config.APPROVAL_THRESHOLD:
            self.status = 'approved'
            self.onboarded_at = datetime.utcnow()
        else:
            self.status = 'rejected'
        
        return self.status
    
    def to_dict(self):
        """Convert to dictionary representation"""
        result = {
            'id': self.id,
            'pid': self.pid,
            'did': self.did,
            'legalName': self.legal_name,
            'type': self.org_type,
            'status': self.status,
            'logo': self.logo,
            'website_url': self.website_url,
            'oa_status': self.oa_status,
            'created_at': self.created_at.isoformat() if self.created_at else None,
            'voting_ends_at': self.voting_ends_at.isoformat() if self.voting_ends_at else None,
            'onboarded_at': self.onboarded_at.isoformat() if self.onboarded_at else None,
            'offboarded_at': self.offboarded_at.isoformat() if self.offboarded_at else None,
            'delegateAuthorityPid': self.delegate_authority_pid,
            'representative': self.representative.to_dict() if self.representative else None
        }
        
        # Add legal identifiers if present
        if self.lei:
            result['lei'] = self.lei
        if self.duns:
            result['duns'] = self.duns
        if self.bic:
            result['bic'] = self.bic
        if self.tin:
            result['tin'] = self.tin
            
        # Add legal address if available
        if self.legal_address:
            result['legalAddress'] = self.legal_address.to_dict()
            
        return result
    
    def __repr__(self):
        return f'<Organization {self.legal_name}>'