# app/models/postal_address.py
from .. import db
import re

class PostalAddress(db.Model):
    __tablename__ = 'postal_addresses'

    id = db.Column(db.Integer, primary_key=True)
    address_line1 = db.Column(db.String(80), nullable=False)     # First line of address (REQUIRED, 1-80 characters)
    address_line2 = db.Column(db.String(80), nullable=True)      # Second line of address (OPTIONAL, 1-80 characters)
    address_line3 = db.Column(db.String(80), nullable=True)      # Third line of address (OPTIONAL, 1-80 characters)
    city = db.Column(db.String(40), nullable=False)              # City (REQUIRED, 1-40 characters)
    region = db.Column(db.String(40), nullable=True)             # Province or region (OPTIONAL, 1-40 characters)
    post_code = db.Column(db.String(40), nullable=False)         # Postal code (REQUIRED, 1-40 characters)
    country = db.Column(db.String(2), nullable=False)            # Country code (REQUIRED, ISO3166 2-letters identifier)
    
    # Foreign key to Organization
    organization_id = db.Column(db.Integer, db.ForeignKey('organizations.id'))
    
    @staticmethod
    def validate_text_field(field_name, value, min_length=1, max_length=80):
        """Validate a text field for Latin characters, numbers, punctuation, and spaces"""
        if not value or len(value) < min_length or len(value) > max_length:
            return False, f"{field_name} must be between {min_length} and {max_length} characters"
            
        # Match Latin characters, numbers, punctuation, and spaces (no newlines)
        # Use r"..." to create a raw string where escapes are ignored
        pattern = r'^[\p{Script=Latin}\p{N}\p{P}\p{Z}]+$'
        try:
            if not re.match(pattern, value, re.UNICODE):
                return False, f"{field_name} must only include Latin letters, numbers, punctuation, and spaces (no newlines)"
        except:
            # Fallback to a simpler pattern if the Unicode regex fails
            if not re.match(r'^[A-Za-z0-9\s\.,\-_\'\"()]+$', value):
                return False, f"{field_name} must only include letters, numbers, spaces and basic punctuation"
                
        return True, None
    
    @staticmethod
    def validate_country_code(value):
        """Validate ISO3166 alpha-2 country code"""
        if not value or len(value) != 2:
            return False, "Country code must be a 2-letter ISO3166 identifier"
            
        # Simple check - should be expanded with actual ISO3166 validation
        if not re.match(r'^[A-Z]{2}$', value):
            return False, "Country code must be a 2-letter ISO3166 identifier"
            
        return True, None
    
    def to_dict(self):
        """Convert to dictionary representation"""
        return {
            'addressLine1': self.address_line1,
            'addressLine2': self.address_line2,
            'addressLine3': self.address_line3,
            'city': self.city,
            'region': self.region,
            'postCode': self.post_code,
            'country': self.country
        }
    
    @classmethod
    def from_dict(cls, data):
        """Create from dictionary data"""
        if not data:
            data = {}
        
        return cls(
            address_line1=data.get('addressLine1', ''),
            address_line2=data.get('addressLine2'),
            address_line3=data.get('addressLine3'),
            city=data.get('city', ''),
            region=data.get('region'),
            post_code=data.get('postCode', ''),
            country=data.get('country', 'IT')  # Default to Italy if not specified
        )