# app/routes/api.py
from flask import Blueprint, jsonify, request, session
from werkzeug.utils import secure_filename
from ..utils.decorators import login_required
from ..services import OrganizationService, VoteService, DocumentService, MarketplaceUserService, AuthService
from ..utils.logo_utils import save_logo, delete_logo
from .. import db
import logging

logger = logging.getLogger(__name__)

api = Blueprint('api', __name__)

@api.route('/organization/<int:organization_id>', methods=['GET'])
@login_required
def get_organization_details(organization_id):
    """Get detailed information about an organization"""
    organization = OrganizationService.get_organization_by_id(organization_id)
    if not organization:
        return jsonify({'error': 'Organization not found'}), 404
    
    # Get document information
    documents = DocumentService.get_organization_documents(organization_id)
    
    # Convert organization to dict
    org_data = organization.to_dict()
    
    # Add document information to the response
    if documents:
        org_data['documents'] = [
            {
                'id': doc.id,
                'filename': doc.filename,
                'uploaded_at': doc.uploaded_at.isoformat() if doc.uploaded_at else None
            } for doc in documents
        ]
    else:
        org_data['documents'] = []
        
    return jsonify(org_data)

@api.route('/postal_address/<int:postal_address_id>', methods=['GET'])
@login_required
def get_postal_address_details(postal_address_id):
    """Get detailed information about an organization"""
    postal_address = OrganizationService.get_postal_address_by_id(postal_address_id)
    if not postal_address:
        return jsonify({'error': 'Postal address not found'}), 404
        
    return jsonify(postal_address.to_dict())

@api.route('/organizations', methods=['GET'])
@login_required
def list_organizations():
    """Get list of organizations with optional filters"""
    status = request.args.get('status')
    industry = request.args.get('industry')
    
    organizations = OrganizationService.get_organizations(
        status=status,
        industry=industry
    )
    
    return jsonify([org.to_dict() for org in organizations])

@api.route('/organizations/<int:organization_id>/votes', methods=['GET'])
@login_required
def get_organization_votes(organization_id):
    """Get voting information for an organization"""
    yes_votes, no_votes = VoteService.get_vote_counts(organization_id)
    
    return jsonify({
        'yes_votes': yes_votes,
        'no_votes': no_votes,
        'total_votes': yes_votes + no_votes
    })

@api.errorhandler(404)
def not_found(error):
    return jsonify({'error': 'Resource not found'}), 404

@api.errorhandler(500)
def internal_server_error(error):
    return jsonify({'error': 'Internal server error'}), 500

@api.route('/document/<int:document_id>/download', methods=['GET'])
@login_required
def download_document(document_id):
    """Download a document"""
    from flask import send_file
    import os
    
    document = DocumentService.get_document_by_id(document_id)
    if not document:
        return jsonify({'error': 'Document not found'}), 404
    
    # Check if file exists
    if not os.path.exists(document.file_path):
        return jsonify({'error': 'File not found'}), 404
    
    # Return the file
    return send_file(
        document.file_path,
        as_attachment=True,
        download_name=document.filename
    )

# ------------------- MarketplaceUser Routes -------------------

@api.route('/marketplace-users', methods=['POST'])
@login_required
def create_marketplace_user():
    """Create a new marketplace user"""
    data = request.json
    
    if not data:
        return jsonify({'error': 'No data provided'}), 400
        
    user, error = MarketplaceUserService.create_marketplace_user(data)
    
    if error:
        return jsonify({'error': error}), 400
        
    return jsonify(user.to_dict()), 201

@api.route('/marketplace-users/<int:user_id>', methods=['GET'])
@login_required
def get_marketplace_user(user_id):
    """Get marketplace user by ID"""
    user = MarketplaceUserService.get_marketplace_user_by_id(user_id)
    
    if not user:
        return jsonify({'error': 'User not found'}), 404
        
    return jsonify(user.to_dict())

@api.route('/marketplace-users/organization/<int:organization_id>', methods=['GET'])
@login_required
def get_marketplace_users_by_organization(organization_id):
    """Get all marketplace users for an organization"""
    users = MarketplaceUserService.get_marketplace_users_by_organization(organization_id)
    
    return jsonify([user.to_dict() for user in users])

@api.route('/marketplace-users/<int:user_id>', methods=['PUT'])
@login_required
def update_marketplace_user(user_id):
    """Update marketplace user details"""
    data = request.json
    
    if not data:
        return jsonify({'error': 'No data provided'}), 400
        
    user, error = MarketplaceUserService.update_marketplace_user(user_id, data)
    
    if error:
        return jsonify({'error': error}), 400
        
    return jsonify(user.to_dict())

@api.route('/marketplace-users/<int:user_id>', methods=['DELETE'])
@login_required
def delete_marketplace_user(user_id):
    """Delete a marketplace user"""
    success, error = MarketplaceUserService.delete_marketplace_user(user_id)
    
    if error:
        return jsonify({'error': error}), 400
        
    return '', 204

@api.route('/marketplace-users/<int:user_id>/sync', methods=['PUT'])
@login_required
def set_marketplace_user_sync(user_id):
    """Set sync status for a marketplace user"""
    data = request.json
    
    if not data or 'sync' not in data:
        return jsonify({'error': 'Sync status not provided'}), 400
        
    success, error = MarketplaceUserService.set_sync_status(user_id, data['sync'])
    
    if error:
        return jsonify({'error': error}), 400
        
    return '', 204

# New routes for integration with external API
@api.route('/marketplace-users/<int:user_id>/offboard', methods=['POST'])
@login_required
def offboard_marketplace_user(user_id):
    """Offboard a marketplace user"""
    success, error = MarketplaceUserService.offboard_marketplace_user(user_id)
    
    if error:
        return jsonify({'error': error}), 400
        
    return '', 204

@api.route('/marketplace-users/<int:user_id>/reinvite', methods=['POST'])
@login_required
def reinvite_marketplace_user(user_id):
    """Reinvite a marketplace user"""
    success, error = MarketplaceUserService.reinvite_marketplace_user(user_id)
    
    if error:
        return jsonify({'error': error}), 400
        
    return '', 204

# ==================== LOGO E WEBSITE ROUTES ====================

@api.route('/organization/update', methods=['POST'])
@login_required
def update_organization_details():
    """Update organization logo and website"""
    try:
        user = AuthService.get_user_by_id(session['user_id'])
        if not user or not user.organization:
            return jsonify({'error': 'Organization not found'}), 404
        
        organization = user.organization
        updated = False
        
        # Handle website URL update
        website_url = request.form.get('website_url', '').strip()
        if website_url:
            # Basic URL validation
            if not website_url.startswith(('http://', 'https://')):
                return jsonify({'error': 'Website URL must start with http:// or https://'}), 400
            
            organization.website_url = website_url
            updated = True
        elif 'website_url' in request.form:  # Empty string means remove website
            organization.website_url = None
            updated = True
        
        # Handle logo upload
        if 'logo' in request.files:
            file = request.files['logo']
            if file and file.filename:
                # Save the new logo
                filename, error = save_logo(file, organization.logo)
                
                if error:
                    return jsonify({'error': error}), 400
                
                organization.logo = filename
                updated = True
        
        if updated:
            db.session.commit()
            logger.info(f"Organization {organization.id} details updated successfully")
            return jsonify({
                'message': 'Organization updated successfully',
                'logo': organization.logo,
                'website_url': organization.website_url
            }), 200
        else:
            return jsonify({'message': 'No changes to update'}), 200
            
    except Exception as e:
        db.session.rollback()
        logger.error(f"Error updating organization: {str(e)}")
        return jsonify({'error': f'Failed to update organization: {str(e)}'}), 500

@api.route('/organization/remove-logo', methods=['POST'])
@login_required
def remove_organization_logo():
    """Remove organization logo"""
    try:
        user = AuthService.get_user_by_id(session['user_id'])
        if not user or not user.organization:
            return jsonify({'error': 'Organization not found'}), 404
        
        organization = user.organization
        
        if not organization.logo:
            return jsonify({'message': 'No logo to remove'}), 200
        
        # Delete the logo file
        old_logo = organization.logo
        delete_logo(old_logo)
        
        # Update database
        organization.logo = None
        db.session.commit()
        
        logger.info(f"Logo removed for organization {organization.id}")
        return jsonify({'message': 'Logo removed successfully'}), 200
        
    except Exception as e:
        db.session.rollback()
        logger.error(f"Error removing logo: {str(e)}")
        return jsonify({'error': f'Failed to remove logo: {str(e)}'}), 500