# app/routes/auth.py 
import logging  # Aggiunto per i log
from flask import Blueprint, request, render_template, redirect, url_for, session, flash, jsonify
from ..services import AuthService, OrganizationService, VoteService, DocumentService, MemberService, MarketplaceUserService
from ..models.marketplace_user import MarketplaceUser
from ..utils.validators import validate_phone_number  # Import the function if it exists in utils.validators
from ..utils.decorators import login_required, nocache
from ..models.organization import FAME_ROA_PID

auth = Blueprint('auth', __name__)

logger = logging.getLogger(__name__)
logging.basicConfig(level=logging.INFO)

@auth.route('/federation', methods=['GET', 'POST'])
@nocache
def federation():
    if 'user_id' in session:
        return redirect(url_for('auth.control_panel'))
    
    # Get the list of active authorities
    logger.info("Fetching active authorities")
    active_authorities = MemberService.get_active_authorities() # Use the new MemberService method
   
    for authority in active_authorities:
        if authority.get('name') == "FAME ROA":
            authority['name'] = "FAME ROOT ONBOARDING AUTHORITY" 
        
    if request.method == 'POST':
        try:
            logger.info("Received federation POST request")
            # Validate passwords match
            if request.form['password'] != request.form['password_confirmation']:
                logger.warning("Passwords do not match")
                return redirect(url_for('auth.federation_feedback',
                    status='error',
                    error_message='Passwords do not match'))
                
            # Get the phone number from the form
            phone_number = request.form.get('phone') or None
            
            # Validate the phone number
            if phone_number:
                logger.info(f"Validating phone number: {phone_number}")
                is_valid, error_message = validate_phone_number(phone_number)
                if not is_valid:
                    logger.warning(f"Phone validation failed: {error_message}")
                    return redirect(url_for('auth.federation_feedback',
                        status='error',
                        error_message=error_message))

            # Create user
            user_data = {
                'first_name': request.form['first_name'],
                'last_name': request.form['last_name'],
                'email': request.form['email'],
                'password': request.form['password'],
                'phone': phone_number
            }
            logger.info(f"Creating user with email: {user_data['email']}")
            user, error = AuthService.create_user(user_data)
            if error:
                logger.error(f"User creation failed: {error}")
                return redirect(url_for('auth.federation_feedback',
                    status='error',
                    error_message=error))
                    
            # After creating the user, inform them about the email confirmation requirement
            flash('A confirmation email has been sent to your email address. Please check your inbox and confirm your email before proceeding.', 'info')

            # Process OA status and related fields
            oa_status = request.form.get('oa_status', 'FAME_ROA')
            did = None
            delegate_authority_pid = None
            
            if oa_status == 'OA':
                # Autonomous OA with own DID
                did = request.form.get('did')
                if not did:
                    logger.warning("Missing DID for OA")
                    return redirect(url_for('auth.federation_feedback',
                        status='error',
                        error_message='DID is required for Autonomous Onboarding Authority'))
            elif oa_status == 'DELEGATE':
                # Delegate to another OA
                delegate_authority_pid = request.form.get('delegate_authority_pid')
                if not delegate_authority_pid:
                    logger.warning("Missing delegate_authority_pid for DELEGATE")
                    return redirect(url_for('auth.federation_feedback',
                        status='error',
                        error_message='Delegate Authority selection is required'))
                    
                # Check if the selected delegate is FAME ROA
                if delegate_authority_pid == FAME_ROA_PID:
                    oa_status = 'FAME_ROA'  # Override the status to FAME_ROA
            
            
            # Create organization with new approach
            legal_address = {
                'addressLine1': request.form['org_address'],
                'city': request.form.get('city', 'Unknown'),  # Add city field to form
                'postCode': request.form.get('post_code', '00000'),  # Add post_code field to form
                'country': request.form.get('country', 'IT')  # Add country field to form or use default
            }
            
            org_data = {
                'org': {
                    'legalName': request.form['org_name'],
                    'legalAddress': legal_address,
                },
                'type': request.form['type'],
                'rep': {
                    'firstName': user.first_name,
                    'lastName': user.last_name,
                    'email': user.email
                },
                'oa_status': oa_status
            }
            
            # NEW: Add logo and website if provided
            if 'logo' in request.files and request.files['logo'].filename:
                org_data['logo'] = request.files['logo']
            if request.form.get('website_url'):
                org_data['website_url'] = request.form.get('website_url')
            
            # Add code here to handle legal identifiers
            # Collect the values of identification fields and add the non-empty ones to org_data
            if request.form.get('lei'):
                org_data['org']['lei'] = request.form.get('lei')
            if request.form.get('duns'):
                org_data['org']['duns'] = request.form.get('duns')
            if request.form.get('bic'):
                org_data['org']['bic'] = request.form.get('bic')
            if request.form.get('tin'):
                org_data['org']['tin'] = request.form.get('tin')

            
                # Note: the VAT field uses the same name 'tin' in the input form,
                # so it does not need to be handled separately

            
            # Add DID or delegateAuthorityPid based on OA status
            if oa_status == 'OA':
                org_data['oa'] = did
            else:
                org_data['oa'] = delegate_authority_pid
            # For FAME_ROA no need to specify anything
            
            # Use the new MemberService to create the organization (call the API)
            organization, error = OrganizationService.create_organization(org_data)
            # Use the OrganizationService to create the organization (old method) TO BE ADDED
            

            if error:
                logger.error(f"Organization creation failed: {error}")
                return redirect(url_for('auth.federation_feedback',
                                        status='error',
                                        error_message=error))


            # Handle document upload (always required)
            if 'authorization_doc' not in request.files or not request.files['authorization_doc']:
                logger.warning("Authorization document is missing")
                return redirect(url_for('auth.federation_feedback',
                    status='error',
                    error_message='Authorization document is required'))

            doc_file = request.files['authorization_doc']
            if doc_file.filename == '':
                logger.warning("No file selected in authorization_doc")
                return redirect(url_for('auth.federation_feedback',
                    status='error',
                    error_message='No file selected'))

            logger.info(f"Saving authorization document for org ID: {organization.id}")
            _, error = DocumentService.save_document(
                request.files['authorization_doc'],
                organization.id
            )
            if error:
                logger.error(f"Error saving document: {error}")
                return redirect(url_for('auth.federation_feedback',
                    status='error',
                    error_message=error))

            logger.info("Federation process completed successfully")
            return redirect(url_for('auth.federation_feedback',
                status='success',
                request_id=organization.id,
                org_name=organization.legal_name,  # Now it is legal_name instead of name
                email=user.email))

        except Exception as e:
            logger.exception("Unhandled exception in federation process")
            return redirect(url_for('auth.federation_feedback',
                status='error',
                error_message=str(e)))

    return render_template('federation.html', active_authorities=active_authorities)

@auth.route('/login', methods=['GET', 'POST'])
@nocache
def login():
    if 'user_id' in session:
        return redirect(url_for('auth.control_panel'))
    
    if request.method == 'POST':
        email = request.form.get('email').lower()
        logger.info(f"Attempting login for: {email}")
        user, error = AuthService.authenticate_user(email, request.form.get('password'))
        
        if error:
            logger.warning(f"Login failed for {email}: {error}")
            return render_template('login.html', error=error)
            
        session['user_id'] = user.id
        session['user_email'] = user.email
        logger.info(f"User {email} logged in successfully")
        return redirect(url_for('auth.control_panel'))
    
    return render_template('login.html')

@auth.route('/control_panel')
@login_required
@nocache
def control_panel():
    user = AuthService.get_user_by_id(session['user_id'])
    if not user:
        logger.warning("User not found in session")
        session.clear()
        return redirect(url_for('auth.login'))
    
    # Check expired votings
    OrganizationService.check_expired_votes()
    
    # Get pending organizations excluding user's own
    pending_organizations = OrganizationService.get_pending_organizations()
    if user.organization:
        pending_organizations = [org for org in pending_organizations 
                               if org.id != user.organization.id]
        
    # Get postal address for user's organization
    postal_address = None
    if user.organization:
        postal_address = user.organization.legal_address
    
    # Get user's votes
    user_votes = {}
    if pending_organizations and user.organization and user.organization.status == 'approved':
        votes = VoteService.get_user_votes(user.id)
        user_votes = {vote.organization_id: vote.vote for vote in votes}
    
    # Get only users of the current organization using the 'aff' filter
    if user.organization and user.organization.pid:
        affiliation_pid = user.organization.pid
        api_users, api_error = MemberService.list_users(filters={'aff': affiliation_pid})
    else:
        # Se l'organizzazione non ha un PID, non mostrare nessun utente o usare un identificativo alternativo
        api_users = []
        api_error = None
        logger.warning(f"Organization {user.organization.legal_name if user.organization else 'Unknown'} has no PID, showing no users")    


    # If there is an error, log it or display an error message    
    if api_error:
       logger.error(f"Error retrieving users from the API: {api_error}")
       api_users = []  # Initialize as an empty list in case of error

    # api_users = [
    #     {
    #         "uid": "u1234567890abcdefghijk1",            
    #         "role": "user",
    #         "affiliation": "org1234567890abcdefghij1",
    #         "lastName": "Bianchi",
    #         "active": True
    #     },
    #     {
    #         "uid": "u1234567890abcdefghijk2", 
    #         "role": "user",
    #         "affiliation": "org1234567890abcdefghij1",
    #         "lastName": "Verdi",
    #         "active": False
    #     }
    # ]

    # get the marketplace users from the local DB for your organization
    local_users = MarketplaceUserService.get_marketplace_users_by_organization(
        user.organization.id if user.organization else None
    )

    # create a local lookup dictionary based on external_uid
    local_lookup = {lu.external_uid: lu for lu in local_users if lu.external_uid}

    # enrich external API data with local data
    enriched_users = []
    for remote_user in api_users:
        uid = remote_user.get("uid")
        local_user = local_lookup.get(uid)

        enriched_users.append({
            "id": uid,
            "role": remote_user.get("role"),
            "affiliation": remote_user.get("affiliation"),
            "last_name": remote_user.get("lastName"),
            "active": remote_user.get("active")            ,
            "first_name": local_user.first_name if local_user else "",
            "email": local_user.email if local_user else "",
            "nickname": local_user.nickname if local_user else "",
            "phone": local_user.phone if local_user else "",
        })
    
    return render_template(
        'control_panel.html',
        user=user,
        organization=user.organization,
        pending_requests=pending_organizations,
        postal_address=postal_address,
        user_votes=user_votes,
        api_users=enriched_users  
    )

# Change from <int:user_id> to <string:user_uid>
@auth.route('/reinvite-user/<string:user_uid>', methods=['POST'])
@login_required
def handle_reinvite_user(user_uid):
    """Endpoint modified to accept external UIDs"""
    try:
        logger.info(f"Processing reinvite for UID {user_uid}")
        
        # Find the user by external_uid instead of ID
        user = MarketplaceUser.query.filter_by(external_uid=user_uid).first()
        if not user:
            return jsonify({"error": "User not found"}), 404
            
        success, error = MarketplaceUserService.reinvite_marketplace_user(user.id)
        
        if not success:
            return jsonify({"error": error}), 400
            
        return jsonify({"success": True}), 200
        
    except Exception as e:
        logger.error(f"Reinvite error: {str(e)}")
        return jsonify({"error": "Internal server error"}), 500

@auth.route('/vote/<int:organization_id>', methods=['POST'])
@login_required
@nocache
def vote(organization_id):
    if 'user_id' not in session:
        return redirect(url_for('auth.login'))
    
    vote_value = request.form.get('vote')
    if vote_value not in ['approve', 'reject']:
        return redirect(url_for('auth.control_panel'))
    
    # Convert vote value to internal format
    vote_value = 'yes' if vote_value == 'approve' else 'no'
    
    # Submit vote through service
    _, error = VoteService.submit_vote(
        organization_id=organization_id,
        voter_id=session['user_id'],
        vote_value=vote_value
    )
    
    if error:
        logger.error(f"Vote submission error: {error}")
        flash(error, 'error')
    
    return redirect(url_for('auth.control_panel'))

@auth.route('/logout')
@nocache
def logout():
    logger.info(f"User {session.get('user_email')} logged out")
    session.clear()
    return redirect(url_for('main.home'))

@auth.route('/federation/feedback')
@nocache
def federation_feedback():
    context = {
        'status': request.args.get('status', 'error'),
        'request_id': request.args.get('request_id'),
        'org_name': request.args.get('org_name'),
        'email': request.args.get('email'),
        'error_message': request.args.get('error_message', 'An unexpected error occurred')
    }
    return render_template('federation_feedback.html', **context)

@auth.route('/confirm-email/<token>')
@nocache
def confirm_email(token):
    """Confirm user's email"""
    success, error = AuthService.confirm_email(token)
    
    if success:
        flash('Your email has been confirmed! You can now log in.', 'success')
        logger.info("Email confirmed successfully")
        return redirect(url_for('auth.login'))
    else:
        # Add detailed error logs
        logger.error(f"Error in email confirmation: {error}")
        flash(error, 'error')
        # Force redirect to login page even in case of error
        # You can remove this line if you prefer to keep the original behavior
        return redirect(url_for('auth.login'))

@auth.route('/resend-confirmation', methods=['GET', 'POST'])
@nocache
def resend_confirmation():
    """Resend confirmation email form"""
    if request.method == 'POST':
        email = request.form.get('email')
        logger.info(f"Resending confirmation email to: {email}")
        success, error = AuthService.resend_confirmation_email(email)
        
        if success:
            flash('A new confirmation email has been sent. Please check your inbox.', 'success')
            return redirect(url_for('auth.login'))
        else:
            logger.error(f"Error resending confirmation email: {error}")
            flash(error, 'error')
    
    return render_template('resend_confirmation.html')

@auth.route('/forgot-password', methods=['GET', 'POST'])
@nocache
def forgot_password():
    """Handle password reset request"""
    if request.method == 'POST':
        email = request.form.get('email')
        if not email:
            logger.warning("Forgot password form submitted with empty email")
            return render_template('forgot_password.html', error="Email is required")
        
        logger.info(f"Password reset requested for: {email}")
        success, error = AuthService.request_password_reset(email)
        if success:
            return render_template('password_reset_sent.html', email=email)
        else:
            logger.error(f"Password reset error for {email}: {error}")
            return render_template('forgot_password.html', error=error)
    
    return render_template('forgot_password.html')

@auth.route('/reset-password/<token>', methods=['GET', 'POST'])
@nocache
def reset_password(token):
    """Handle password reset with token"""
    if request.method == 'POST':
        password = request.form.get('password')
        password_confirmation = request.form.get('password_confirmation')
        
        if not password:
            logger.warning("Password reset form submitted with empty password")
            return render_template('reset_password.html', 
                                 token=token, 
                                 error="Password is required")
                                 
        if password != password_confirmation:
            logger.warning("Password reset mismatch")
            return render_template('reset_password.html', 
                                 token=token, 
                                 error="Passwords do not match")
        
        success, error = AuthService.reset_password(token, password)
        if success:
            logger.info("Password reset successful")
            flash('Your password has been reset successfully. You can now log in with your new password.', 'success')
            return redirect(url_for('auth.login'))
        else:
            logger.error(f"Password reset error: {error}")
            return render_template('reset_password.html', 
                                 token=token, 
                                 error=error)
    
    return render_template('reset_password.html', token=token)
