# app/routes/internal_api.py
from datetime import datetime
import re
from flask import Blueprint, jsonify, request, Response, g
import requests
import json
from functools import wraps
from ..config import Config
from .. import db
from ..models import User, Organization, PostalAddress, Vote, Document, MarketplaceUser

from werkzeug.utils import secure_filename
from ..utils.logo_utils import save_logo, delete_logo
import os


internal_api = Blueprint('internal_api', __name__)

# Basic authentication configuration
API_USERNAME = Config.INTERNAL_API_USERNAME
API_PASSWORD = Config.INTERNAL_API_PASSWORD  # In production, use Config or environment variables

# Authentication decorator
def require_auth(f):
    @wraps(f)
    def decorated(*args, **kwargs):
        auth = request.authorization
        if not auth or auth.username != API_USERNAME or auth.password != API_PASSWORD:
            return jsonify({"error": "Unauthorized access"}), 401
        return f(*args, **kwargs)
    return decorated

# Universal helper function to forward any type of request
def forward_request(method, url, params=None, data=None, headers=None, timeout=30):
    """
    Forwards a generic request to a specified URL
    
    Args:
        method (str): HTTP method (GET, POST, PUT, DELETE, etc.)
        url (str): Full URL to forward the request to
        params (dict): Optional query string parameters
        data (dict/str): Data to send in the body (for POST/PUT)
        headers (dict): Custom HTTP headers
        timeout (int): Timeout in seconds
    
    Returns:
        tuple: (response_data, status_code, response_headers)
    """
    try:
        # Log the request we are about to send
        print(f"Forwarding {method} request to: {url}")
        if params:
            print(f"With params: {params}")
        if data:
            print(f"With data: {data}")
        
        # If no custom headers are specified, use default ones
        if not headers:
            headers = {'Content-Type': 'application/json'}
        
        # Send the request to the specified URL with the specified method
        response = requests.request(
            method=method.upper(),
            url=url,
            params=params,
            data=json.dumps(data) if isinstance(data, dict) else data,
            headers=headers,
            timeout=timeout
        )
        
        # Try to interpret the response as JSON
        try:
            response_data = response.json()
        except ValueError:
            # If the response is not JSON, return the text as a string
            response_data = {"message": response.text}
        
        # Extract relevant headers from the response
        response_headers = {k: v for k, v in response.headers.items()}
        
        return response_data, response.status_code, response_headers
        
    except requests.RequestException as e:
        # Handle connection errors
        error_message = f"Error connecting to API: {str(e)}"
        print(error_message)
        return {"error": error_message}, 502, {}  # Bad Gateway

# Universal proxy route
@internal_api.route('/api/proxy', methods=['POST'])
@require_auth
def universal_proxy():
    """
    Universal proxy that accepts any type of request in the body
    
    Example body:
    {
        "method": "GET|POST|PUT|DELETE",
        "url": "http://service-name:port/path",  // Full URL
        "params": {"param1": "value1"},          // optional
        "data": {...},                           // optional, for POST/PUT
        "headers": {...}                         // optional, custom headers
    }
    """
    try:
        # Verify that the request body is JSON
        if not request.is_json:
            return jsonify({"error": "Request body must be JSON"}), 400
        
        # Extract parameters from the body
        proxy_request = request.get_json()
        
        # Verify that required parameters are present
        if 'method' not in proxy_request:
            return jsonify({"error": "Missing required parameter: method"}), 400
        if 'url' not in proxy_request:
            return jsonify({"error": "Missing required parameter: url"}), 400
            
        # Extract parameters
        method = proxy_request.get('method').upper()
        url = proxy_request.get('url')
        params = proxy_request.get('params')
        data = proxy_request.get('data')
        headers = proxy_request.get('headers')
        
        # Verify that the method is valid
        valid_methods = ['GET', 'POST', 'PUT', 'DELETE', 'PATCH', 'HEAD', 'OPTIONS']
        if method not in valid_methods:
            return jsonify({"error": f"Invalid method: {method}. Must be one of: {', '.join(valid_methods)}"}), 400
        
        # URL validation (basic security check)
        if not url.startswith(('http://', 'https://')):
            return jsonify({"error": "URL must start with http:// or https://"}), 400
        
        # Forward the request to the specified URL
        response_data, status_code, response_headers = forward_request(
            method=method,
            url=url,
            params=params,
            data=data,
            headers=headers
        )
        
        # Create the complete response with request details
        response = {
            "data": response_data,
            "status": status_code,
            "request": {
                "method": method,
                "url": url,
                "params": params,
                "headers": headers if headers else {},
                # Do not include sent data for security
                "data_sent": True if data else False
            },
            "response_headers": response_headers
        }
        
        return jsonify(response), status_code
        
    except Exception as e:
        # Generic error handling
        error_message = f"Error processing proxy request: {str(e)}"
        print(error_message)
        return jsonify({"error": error_message}), 500

# Route to check if the API is active
@internal_api.route('/api/health', methods=['GET'])
def health_check():
    """Route to check if the internal API is running"""
    return jsonify({
        "status": "ok",
        "message": "FAME Internal API is running"
    }), 200

# ----- CRUD API for models ----- #

# ----- User CRUD ----- #

@internal_api.route('/api/users', methods=['GET'])
@require_auth
def get_users():
    """Retrieve all users"""
    try:
        users = User.query.all()
        return jsonify([user.to_dict() for user in users]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/users/<int:user_id>', methods=['GET'])
@require_auth
def get_user(user_id):
    """Retrieve a specific user"""
    try:
        user = User.query.get(user_id)
        if not user:
            return jsonify({"error": "User not found"}), 404
        return jsonify(user.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/users', methods=['POST'])
@require_auth
def create_user():
    """Create a new user"""
    try:
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify required fields
        required_fields = ['first_name', 'last_name', 'email', 'password']
        for field in required_fields:
            if field not in data:
                return jsonify({"error": f"Missing required field: {field}"}), 400
        
        # Check if the email is already in use
        if User.query.filter_by(email=data['email']).first():
            return jsonify({"error": "Email already in use"}), 409
        
        # Create new user
        user = User(
            first_name=data['first_name'],
            last_name=data['last_name'],
            email=data['email'],
            password=data['password'],  # Password setter applies hashing
            phone=data.get('phone'),
            email_confirmed=data.get('email_confirmed', False)
        )
        
        db.session.add(user)
        db.session.commit()
        
        return jsonify(user.to_dict()), 201
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/users/<int:user_id>', methods=['PUT'])
@require_auth
def update_user(user_id):
    """Update a user"""
    try:
        user = User.query.get(user_id)
        if not user:
            return jsonify({"error": "User not found"}), 404
            
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Update fields if present
        if 'first_name' in data:
            user.first_name = data['first_name']
        if 'last_name' in data:
            user.last_name = data['last_name']
        if 'email' in data:
            # Check if the new email is already in use by another user
            existing = User.query.filter_by(email=data['email']).first()
            if existing and existing.id != user_id:
                return jsonify({"error": "Email already in use"}), 409
            user.email = data['email']
        if 'password' in data:
            user.password = data['password']  # Setter applies hashing
        if 'phone' in data:
            user.phone = data['phone']
        if 'email_confirmed' in data:
            user.email_confirmed = data['email_confirmed']
        
        db.session.commit()
        
        return jsonify(user.to_dict()), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/users/<int:user_id>', methods=['DELETE'])
@require_auth
def delete_user(user_id):
    """Delete a user"""
    try:
        user = User.query.get(user_id)
        if not user:
            return jsonify({"error": "User not found"}), 404
            
        db.session.delete(user)
        db.session.commit()
        
        return jsonify({"message": "User deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ----- Organization CRUD ----- #

@internal_api.route('/api/organizations', methods=['GET'])
@require_auth
def get_organizations():
    """Retrieve all organizations"""
    try:
        organizations = Organization.query.all()
        return jsonify([org.to_dict() for org in organizations]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>', methods=['GET'])
@require_auth
def get_organization(org_id):
    """Retrieve a specific organization"""
    try:
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
        return jsonify(org.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations', methods=['POST'])
@require_auth
def create_organization():
    """Create a new organization"""
    try:
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify required fields
        required_fields = ['legal_name', 'org_type', 'representative_id']
        for field in required_fields:
            if field not in data:
                return jsonify({"error": f"Missing required field: {field}"}), 400
        
        # Verify that the representative exists
        representative = User.query.get(data['representative_id'])
        if not representative:
            return jsonify({"error": "Representative not found"}), 404
        
        # Create new organization
        org = Organization(
            legal_name=data['legal_name'],
            org_type=data['org_type'],
            representative_id=data['representative_id'],
            oa_status=data.get('oa_status', 'FAME_ROA'),
            delegate_authority_pid=data.get('delegate_authority_pid'),
            did=data.get('did'),
            status=data.get('status', 'pending'),
            lei=data.get('lei'),
            duns=data.get('duns'),
            bic=data.get('bic'),
            tin=data.get('tin')
        )
        
        db.session.add(org)
        db.session.flush()  # To get the ID
        
        # If address data is included, create the associated address
        if 'address' in data and isinstance(data['address'], dict):
            addr_data = data['address']
            address = PostalAddress(
                organization_id=org.id,
                address_line1=addr_data.get('address_line1'),
                address_line2=addr_data.get('address_line2'),
                address_line3=addr_data.get('address_line3'),
                city=addr_data.get('city'),
                region=addr_data.get('region'),
                post_code=addr_data.get('post_code'),
                country=addr_data.get('country', 'IT')
            )
            db.session.add(address)
        
        db.session.commit()
        
        return jsonify(org.to_dict()), 201
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>', methods=['PUT'])
@require_auth
def update_organization(org_id):
    """Update an organization"""
    try:
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
            
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Update fields if present
        if 'legal_name' in data:
            org.legal_name = data['legal_name']
        # Adding the possibility to update the PID
        if 'pid' in data:
            # First, verify that the new PID is not already in use by another organization
            existing_org = Organization.query.filter_by(pid=data['pid']).first()
            if existing_org and existing_org.id != org_id:
                return jsonify({"error": "The provided PID is already in use by another organization"}), 409

            # Log the change for traceability
            print(f"Updating PID of organization {org.id} from '{org.pid}' to '{data['pid']}'")

            # Update the PID
            org.pid = data['pid']
        if 'org_type' in data:
            org.org_type = data['org_type']
        if 'oa_status' in data:
            org.oa_status = data['oa_status']
        if 'delegate_authority_pid' in data:
            org.delegate_authority_pid = data['delegate_authority_pid']
        if 'did' in data:
            org.did = data['did']
        if 'status' in data:
            org.status = data['status']
        if 'lei' in data:
            org.lei = data['lei']
        if 'duns' in data:
            org.duns = data['duns']
        if 'bic' in data:
            org.bic = data['bic']
        if 'tin' in data:
            org.tin = data['tin']
        if 'representative_id' in data:
            # Verify that the new representative exists
            representative = User.query.get(data['representative_id'])
            if not representative:
                return jsonify({"error": "Representative not found"}), 404
            org.representative_id = data['representative_id']

        if 'onboarded_at' in data:
            val = data['onboarded_at']
            if val is None:
                org.onboarded_at = None
            elif isinstance(val, str):
                s = val.strip()
                # convert trailing Z to +00:00
                if s.endswith('Z'):
                    s = s[:-1] + '+00:00'

                # Try direct parsing first (accepts both 'T' and space as separators)
                try:
                    org.onboarded_at = datetime.fromisoformat(s)
                except Exception:
                    # If direct parsing fails (e.g. month/day not zero-padded) pad them and retry
                    s2 = re.sub(r'^(\d{4})-(\d{1,2})-(\d{1,2})',
                                lambda m: f"{m.group(1)}-{int(m.group(2)):02d}-{int(m.group(3)):02d}",
                                s)
                    try:
                        org.onboarded_at = datetime.fromisoformat(s2)
                    except Exception:
                        return jsonify({"error": "Invalid datetime format for onboarded_at. Use ISO (e.g. 2025-02-27T15:45:00) or 'YYYY-MM-DD HH:MM:SS'"}), 400
            else:
                return jsonify({"error": "Invalid type for onboarded_at; must be an ISO string or null"}), 400       
            
        # Update the address if present
        if 'address' in data and isinstance(data['address'], dict):
            addr_data = data['address']
            # Find the existing address or create a new one
            address = PostalAddress.query.filter_by(organization_id=org.id).first()
            if not address:
                address = PostalAddress(organization_id=org.id)
                db.session.add(address)
            
            # Update address fields
            if 'address_line1' in addr_data:
                address.address_line1 = addr_data['address_line1']
            if 'address_line2' in addr_data:
                address.address_line2 = addr_data['address_line2']
            if 'address_line3' in addr_data:
                address.address_line3 = addr_data['address_line3']
            if 'city' in addr_data:
                address.city = addr_data['city']
            if 'region' in addr_data:
                address.region = addr_data['region']
            if 'post_code' in addr_data:
                address.post_code = addr_data['post_code']
            if 'country' in addr_data:
                address.country = addr_data['country']
        
        db.session.commit()
        
        return jsonify(org.to_dict()), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>', methods=['DELETE'])
@require_auth
def delete_organization(org_id):
    """Delete an organization"""
    try:
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
            
        db.session.delete(org)
        db.session.commit()
        
        return jsonify({"message": "Organization deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500
    
# ----- MarketplaceUser CRUD ----- #

@internal_api.route('/api/marketplace-users', methods=['GET'])
@require_auth
def get_marketplace_users():
    """Retrieve all marketplace users"""
    try:
        users = MarketplaceUser.query.all()
        return jsonify([user.to_dict() for user in users]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>/marketplace-users', methods=['GET'])
@require_auth
def get_marketplace_users_for_organization(org_id):
    """Retrieve all marketplace users for a specific organization"""
    try:
        users = MarketplaceUser.query.filter_by(organization_id=org_id).all()
        return jsonify([user.to_dict() for user in users]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/marketplace-users/<int:user_id>', methods=['GET'])
@require_auth
def get_marketplace_user(user_id):
    """Retrieve a specific marketplace user"""
    try:
        user = MarketplaceUser.query.get(user_id)
        if not user:
            return jsonify({"error": "Marketplace user not found"}), 404
        return jsonify(user.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/marketplace-users', methods=['POST'])
@require_auth
def create_marketplace_user():
    """Create a new marketplace user"""
    try:
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify required fields
        required_fields = ['first_name', 'last_name', 'email', 'organization_id']
        for field in required_fields:
            if field not in data:
                return jsonify({"error": f"Missing required field: {field}"}), 400
        
        # Verify that the organization exists
        organization = Organization.query.get(data['organization_id'])
        if not organization:
            return jsonify({"error": "Organization not found"}), 404
            
        # Verify unique email for the organization
        existing = MarketplaceUser.query.filter_by(
            email=data['email'],
            organization_id=data['organization_id']
        ).first()
        if existing:
            return jsonify({"error": "Email already exists for this organization"}), 409
            
        # Verify nickname length
        if 'nickname' in data and data['nickname'] and len(data['nickname']) > 15:
            return jsonify({"error": "Nickname cannot exceed 15 characters"}), 400
        
        # Create new marketplace user
        marketplace_user = MarketplaceUser(
            first_name=data['first_name'],
            last_name=data['last_name'],
            email=data['email'],
            organization_id=data['organization_id'],
            nickname=data.get('nickname'),
            phone=data.get('phone'),
            role=data.get('role', 'user'),
            sync=data.get('sync', False),
            status=data.get('status', 'invited'),
            external_uid=data.get('external_uid')
        )
        
        db.session.add(marketplace_user)
        db.session.commit()
        
        return jsonify(marketplace_user.to_dict()), 201
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/marketplace-users/<int:user_id>', methods=['PUT'])
@require_auth
def update_marketplace_user(user_id):
    """Update a marketplace user"""
    try:
        user = MarketplaceUser.query.get(user_id)
        if not user:
            return jsonify({"error": "Marketplace user not found"}), 404
            
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify unique email for the organization
        if 'email' in data and data['email'] != user.email:
            existing = MarketplaceUser.query.filter_by(
                email=data['email'],
                organization_id=user.organization_id
            ).first()
            if existing:
                return jsonify({"error": "Email already exists for this organization"}), 409
                
        # Verify nickname length
        if 'nickname' in data and data['nickname'] and len(data['nickname']) > 15:
            return jsonify({"error": "Nickname cannot exceed 15 characters"}), 400
        
        # Update fields if present
        if 'first_name' in data:
            user.first_name = data['first_name']
        if 'last_name' in data:
            user.last_name = data['last_name']
        if 'email' in data:
            user.email = data['email']
        if 'nickname' in data:
            user.nickname = data['nickname']
        if 'phone' in data:
            user.phone = data['phone']
        if 'role' in data:
            user.role = data['role']
        if 'sync' in data:
            user.sync = data['sync']
        if 'status' in data:
            user.status = data['status']
        if 'external_uid' in data:
            user.external_uid = data['external_uid']
        
        db.session.commit()
        
        return jsonify(user.to_dict()), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/marketplace-users/<int:user_id>', methods=['DELETE'])
@require_auth
def delete_marketplace_user(user_id):
    """Delete a marketplace user"""
    try:
        user = MarketplaceUser.query.get(user_id)
        if not user:
            return jsonify({"error": "Marketplace user not found"}), 404
            
        db.session.delete(user)
        db.session.commit()
        
        return jsonify({"message": "Marketplace user deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ----- PostalAddress CRUD ----- #
@internal_api.route('/api/addresses', methods=['GET'])
@require_auth
def get_addresses():
    """Retrieve all postal addresses"""
    try:
        addresses = PostalAddress.query.all()
        return jsonify([addr.to_dict() for addr in addresses]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/addresses/<int:addr_id>', methods=['GET'])
@require_auth
def get_address(addr_id):
    """Retrieve a specific postal address"""
    try:
        address = PostalAddress.query.get(addr_id)
        if not address:
            return jsonify({"error": "Address not found"}), 404
        return jsonify(address.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>/address', methods=['GET'])
@require_auth
def get_organization_address(org_id):
    """Retrieve the postal address for a specific organization"""
    try:
        address = PostalAddress.query.filter_by(organization_id=org_id).first()
        if not address:
            return jsonify({"error": "Address not found"}), 404
        return jsonify(address.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/addresses', methods=['POST'])
@require_auth
def create_address():
    """Create a new postal address"""
    try:
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify required fields
        required_fields = ['organization_id', 'address_line1', 'city', 'post_code', 'country']
        for field in required_fields:
            if field not in data:
                return jsonify({"error": f"Missing required field: {field}"}), 400
        
        # Verify that the organization exists
        organization = Organization.query.get(data['organization_id'])
        if not organization:
            return jsonify({"error": "Organization not found"}), 404
        
        # Verify if the organization already has an address
        existing = PostalAddress.query.filter_by(organization_id=data['organization_id']).first()
        if existing:
            return jsonify({"error": "Organization already has an address"}), 409
        
        # Create a new address
        address = PostalAddress(
            organization_id=data['organization_id'],
            address_line1=data['address_line1'],
            address_line2=data.get('address_line2'),
            address_line3=data.get('address_line3'),
            city=data['city'],
            region=data.get('region'),
            post_code=data['post_code'],
            country=data['country']
        )
        
        db.session.add(address)
        db.session.commit()
        
        return jsonify(address.to_dict()), 201
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/addresses/<int:addr_id>', methods=['PUT'])
@require_auth
def update_address(addr_id):
    """Update a postal address"""
    try:
        address = PostalAddress.query.get(addr_id)
        if not address:
            return jsonify({"error": "Address not found"}), 404
            
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Update fields if present
        if 'address_line1' in data:
            address.address_line1 = data['address_line1']
        if 'address_line2' in data:
            address.address_line2 = data['address_line2']
        if 'address_line3' in data:
            address.address_line3 = data['address_line3']
        if 'city' in data:
            address.city = data['city']
        if 'region' in data:
            address.region = data['region']
        if 'post_code' in data:
            address.post_code = data['post_code']
        if 'country' in data:
            address.country = data['country']
        
        db.session.commit()
        
        return jsonify(address.to_dict()), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/addresses/<int:addr_id>', methods=['DELETE'])
@require_auth
def delete_address(addr_id):
    """Delete a postal address"""
    try:
        address = PostalAddress.query.get(addr_id)
        if not address:
            return jsonify({"error": "Address not found"}), 404
            
        db.session.delete(address)
        db.session.commit()
        
        return jsonify({"message": "Address deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ----- Vote CRUD ----- #

@internal_api.route('/api/votes', methods=['GET'])
@require_auth
def get_votes():
    """Retrieve all votes"""
    try:
        votes = Vote.query.all()
        return jsonify([vote.to_dict() for vote in votes]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>/votes', methods=['GET'])
@require_auth
def get_votes_for_organization(org_id):
    """Retrieve all votes for a specific organization"""
    try:
        votes = Vote.query.filter_by(organization_id=org_id).all()
        return jsonify([vote.to_dict() for vote in votes]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/votes', methods=['POST'])
@require_auth
def create_vote():
    """Cast a vote for an organization"""
    try:
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify required fields
        required_fields = ['organization_id', 'voter_id', 'vote']
        for field in required_fields:
            if field not in data:
                return jsonify({"error": f"Missing required field: {field}"}), 400
        
        # Verify that the voter and organization exist
        voter = User.query.get(data['voter_id'])
        if not voter:
            return jsonify({"error": "Voter not found"}), 404
            
        org = Organization.query.get(data['organization_id'])
        if not org:
            return jsonify({"error": "Organization not found"}), 404
            
        # Verify that the organization is in pending status
        if org.status != 'pending':
            return jsonify({"error": "Organization must be in 'pending' status to receive votes"}), 400
            
        # Verify that the vote is valid
        if data['vote'] not in ['yes', 'no']:
            return jsonify({"error": "Vote must be 'yes' or 'no'"}), 400
            
        # Check if the user has already voted for this organization
        existing_vote = Vote.query.filter_by(
            organization_id=data['organization_id'],
            voter_id=data['voter_id']
        ).first()
        
        if existing_vote:
            # Update the existing vote
            existing_vote.vote = data['vote']
            db.session.commit()
            return jsonify(existing_vote.to_dict()), 200
        else:
            # Create a new vote
            vote = Vote(
                organization_id=data['organization_id'],
                voter_id=data['voter_id'],
                vote=data['vote']
            )
            
            db.session.add(vote)
            db.session.commit()
            
            return jsonify(vote.to_dict()), 201
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/votes/<int:vote_id>', methods=['PUT'])
@require_auth
def update_vote(vote_id):
    """Update a vote"""
    try:
        vote = Vote.query.get(vote_id)
        if not vote:
            return jsonify({"error": "Vote not found"}), 404
            
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
            
        data = request.get_json()
        
        # Verify that the vote is valid
        if 'vote' in data:
            if data['vote'] not in ['yes', 'no']:
                return jsonify({"error": "Vote must be 'yes' or 'no'"}), 400
            vote.vote = data['vote']
        
        db.session.commit()
        
        return jsonify(vote.to_dict()), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/votes/<int:vote_id>', methods=['DELETE'])
@require_auth
def delete_vote(vote_id):
    """Delete a vote"""
    try:
        vote = Vote.query.get(vote_id)
        if not vote:
            return jsonify({"error": "Vote not found"}), 404
            
        db.session.delete(vote)
        db.session.commit()
        
        return jsonify({"message": "Vote deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ----- Document CRUD ----- #

@internal_api.route('/api/documents', methods=['GET'])
@require_auth
def get_documents():
    """Retrieve all documents"""
    try:
        documents = Document.query.all()
        return jsonify([doc.to_dict() for doc in documents]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/organizations/<int:org_id>/documents', methods=['GET'])
@require_auth
def get_documents_for_organization(org_id):
    """Retrieve all documents for a specific organization"""
    try:
        documents = Document.query.filter_by(organization_id=org_id).all()
        return jsonify([doc.to_dict() for doc in documents]), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/documents/<int:doc_id>', methods=['GET'])
@require_auth
def get_document(doc_id):
    """Retrieve a specific document's metadata"""
    try:
        document = Document.query.get(doc_id)
        if not document:
            return jsonify({"error": "Document not found"}), 404
        return jsonify(document.to_dict()), 200
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/documents/<int:doc_id>/download', methods=['GET'])
@require_auth
def download_document(doc_id):
    """Download a document file"""
    try:
        from flask import send_file
        import os
        
        document = Document.query.get(doc_id)
        if not document:
            return jsonify({"error": "Document not found"}), 404
            
        # Verify that the file exists
        if not os.path.exists(document.file_path):
            return jsonify({"error": "File not found"}), 404
            
        # Send the file
        return send_file(
            document.file_path,
            as_attachment=True,
            download_name=document.filename
        )
    except Exception as e:
        return jsonify({"error": str(e)}), 500

@internal_api.route('/api/documents/<int:doc_id>', methods=['DELETE'])
@require_auth
def delete_document(doc_id):
    """Delete a document"""
    try:
        document = Document.query.get(doc_id)
        if not document:
            return jsonify({"error": "Document not found"}), 404
            
        # Delete the associated file
        import os
        if os.path.exists(document.file_path):
            os.remove(document.file_path)
            
        db.session.delete(document)
        db.session.commit()
        
        return jsonify({"message": "Document deleted successfully"}), 200
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500

# ==================== API PER UPLOAD LOGO ====================

@internal_api.route('/api/organizations/<int:org_id>/logo', methods=['POST'])
@require_auth
def upload_organization_logo(org_id):
    """
    Upload logo for an organization
    Accepts multipart/form-data with:
    - logo: file (required)
    - website_url: string (optional)
    """
    try:
        # Get organization
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
        
        # Check if logo file is present
        if 'logo' not in request.files:
            return jsonify({"error": "No logo file provided"}), 400
        
        logo_file = request.files['logo']
        
        if logo_file.filename == '':
            return jsonify({"error": "No file selected"}), 400
        
        # Save the logo (handles validation, naming, old file deletion)
        filename, error = save_logo(logo_file, org.logo)
        
        if error:
            return jsonify({"error": error}), 400
        
        # Update organization with new logo
        org.logo = filename
        
        # Also update website_url if provided
        website_url = request.form.get('website_url')
        if website_url:
            # Basic URL validation
            if not website_url.startswith(('http://', 'https://')):
                # Rollback logo save if URL is invalid
                delete_logo(filename)
                return jsonify({"error": "Website URL must start with http:// or https://"}), 400
            
            org.website_url = website_url
        
        db.session.commit()
        
        return jsonify({
            "message": "Logo uploaded successfully",
            "organization": org.to_dict()
        }), 200
        
    except Exception as e:
        db.session.rollback()
        # Try to cleanup uploaded file if database commit failed
        if 'filename' in locals():
            delete_logo(filename)
        return jsonify({"error": str(e)}), 500


@internal_api.route('/api/organizations/<int:org_id>/logo', methods=['DELETE'])
@require_auth
def delete_organization_logo(org_id):
    """
    Delete logo for an organization
    """
    try:
        # Get organization
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
        
        if not org.logo:
            return jsonify({"message": "No logo to delete"}), 200
        
        # Delete the file
        old_logo = org.logo
        delete_logo(old_logo)
        
        # Update database
        org.logo = None
        db.session.commit()
        
        return jsonify({
            "message": "Logo deleted successfully",
            "organization": org.to_dict()
        }), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500


@internal_api.route('/api/organizations/<int:org_id>/website', methods=['PUT'])
@require_auth
def update_organization_website(org_id):
    """
    Update website URL for an organization
    Accepts JSON with:
    - website_url: string (required, can be empty to remove)
    """
    try:
        # Get organization
        org = Organization.query.get(org_id)
        if not org:
            return jsonify({"error": "Organization not found"}), 404
        
        if not request.is_json:
            return jsonify({"error": "Request must be JSON"}), 400
        
        data = request.get_json()
        website_url = data.get('website_url', '').strip()
        
        if website_url:
            # Validate URL
            if not website_url.startswith(('http://', 'https://')):
                return jsonify({"error": "Website URL must start with http:// or https://"}), 400
            
            org.website_url = website_url
        else:
            # Empty string means remove website
            org.website_url = None
        
        db.session.commit()
        
        return jsonify({
            "message": "Website updated successfully",
            "organization": org.to_dict()
        }), 200
        
    except Exception as e:
        db.session.rollback()
        return jsonify({"error": str(e)}), 500
    
# ==================== TEST: list trading accounts via proxy ====================

@internal_api.route('/api/test/accounts', methods=['GET'])
@require_auth
def test_list_trading_accounts():
    """
    Test helper to call GET /gov/v1.0/accounts on the GOV service
    via the internal proxy, so you can call it easily from Postman.

    Query params (all optional):
    - usr: owner UID (user)
    - own: owner PID (affiliation)
    - aut: onboarding authority DID
    - sta: status (ACTIVE, INACTIVE, TERMINATED)
    """
    try:
        # Collect parameters from the query string
        usr = request.args.get('usr')
        own = request.args.get('own')
        aut = request.args.get('aut')
        sta = request.args.get('sta')

        # Build the params dict only with the ones present
        params = {}
        if usr:
            params['usr'] = usr
        if own:
            params['own'] = own
        if aut:
            params['aut'] = aut
        if sta:
            params['sta'] = sta

        # Prepare the payload for the universal proxy
        proxy_body = {
            "method": "GET",
            "url": "http://gov-internal:7008/gov/v1.0/accounts",
            "params": params,
            "headers": {
                "Accept": "application/json",
                "Authorization": f"Bearer {Config.FAME_API_KEY}"
            }
        }

        # Call the internal proxy as you would from Postman
        # (reuse forward_request instead of making a new HTTP request)
        response_data, status_code, response_headers = forward_request(
            method=proxy_body["method"],
            url=proxy_body["url"],
            params=proxy_body.get("params"),
            data=None,
            headers=proxy_body.get("headers"),
        )

        # Return the result directly to Postman
        return jsonify({
            "upstream_status": status_code,
            "upstream_headers": response_headers,
            "upstream_data": response_data,
            "request_params": params,
        }), status_code

    except Exception as e:
        return jsonify({"error": f"Error calling GOV accounts API: {str(e)}"}), 500
