# app/routes/public_api.py
from flask import Blueprint, jsonify, request, after_this_request
from ..models import Organization, MarketplaceUser
from .. import db

public_api = Blueprint('public_api', __name__)

@public_api.route('/federation-stats', methods=['GET'])
def get_federation_stats():
    """Get public statistics and basic information about the federation"""
    
    # Handle CORS for this endpoint
    @after_this_request
    def add_cors_headers(response):
        # Allow specific origin
        response.headers['Access-Control-Allow-Origin'] = 'https://marketplace.fame-horizon.eu'
        # Allow credentials (cookies, authorization headers)
        response.headers['Access-Control-Allow-Credentials'] = 'true'
        # Allow specific methods
        response.headers['Access-Control-Allow-Methods'] = 'GET, OPTIONS'
        # Allow specific headers
        response.headers['Access-Control-Allow-Headers'] = 'Content-Type, Authorization'
        return response
        
    try:
        # Count approved organizations (federated organizations)
        federated_orgs = Organization.query.filter_by(
            status='approved',
            offboarded_at=None
        ).all()
        
        federated_orgs_count = len(federated_orgs)
        
        # Count marketplace users
        marketplace_users_count = MarketplaceUser.query.count()
        
        # Prepare organization list with limited information
        organizations_list = []
        for org in federated_orgs:
            organizations_list.append({
                'legal_name': org.legal_name,
                'org_type': org.org_type,
                'onboarded_at': org.onboarded_at.isoformat() if org.onboarded_at else None
            })
        
        return jsonify({
            "federated_organizations_count": federated_orgs_count,
            "marketplace_users_count": marketplace_users_count,
            "organizations": organizations_list
        }), 200
    except Exception as e:
        print(f"Error in federation stats API: {str(e)}")
        return jsonify({"error": "An error occurred while retrieving federation statistics"}), 500

# Handle OPTIONS requests for CORS preflight
@public_api.route('/federation-stats', methods=['OPTIONS'])
def options_federation_stats():
    response = jsonify({'status': 'OK'})
    response.headers['Access-Control-Allow-Origin'] = 'https://marketplace.fame-horizon.eu'
    response.headers['Access-Control-Allow-Credentials'] = 'true'
    response.headers['Access-Control-Allow-Methods'] = 'GET, OPTIONS'
    response.headers['Access-Control-Allow-Headers'] = 'Content-Type, Authorization'
    return response