# app/routes/wallet.py
"""
Wallet Routes - Digital Euro + FAME Marketplace integration
"""

from flask import Blueprint, render_template, request, jsonify, flash, redirect, url_for, session
from app.models.digital_euro_account import DigitalEuroAccount, WalletTransaction
from app.models.organization import Organization
from app.models.user import User
from app.services.digital_euro_service import digital_euro_service
from app.services.marketplace_wallet_service import marketplace_wallet_service
from app.utils.decorators import login_required
from app import db
import logging

logger = logging.getLogger(__name__)

bp = Blueprint('wallet', __name__, url_prefix='/wallet')


def get_current_user():
    """Get current logged-in user from session"""
    if 'user_id' not in session:
        return None
    return User.query.get(session['user_id'])


def get_current_organization():
    """Get organization from current logged-in user"""
    user = get_current_user()
    if not user:
        return None
    return user.organization


@bp.route('/')
@login_required
def index():
    """Marketplace Wallet Gateway - Always shows first"""
    user = get_current_user()
    organization = get_current_organization()
    
    if not organization:
        flash('Your account is not associated with any organization', 'warning')
        return redirect(url_for('auth.control_panel'))
    
    de_account = DigitalEuroAccount.query.filter_by(
        organization_id=organization.id,
        is_active=True
    ).first()
    
    return render_template(
        'wallet_gateway.html',
        organization=organization,
        de_account=de_account,
        user=user
    )


@bp.route('/dashboard')
@login_required
def dashboard():
    """Marketplace Wallet Dashboard - Only accessible with DE account"""
    user = get_current_user()
    organization = get_current_organization()
    
    if not organization:
        flash('Your account is not associated with any organization', 'warning')
        return redirect(url_for('auth.control_panel'))
    
    # CHECK: Redirect if no account
    de_account = DigitalEuroAccount.query.filter_by(
        organization_id=organization.id,
        is_active=True
    ).first()
    
    if not de_account:
        flash('Please create a Digital Euro account first', 'warning')
        return redirect(url_for('wallet.index'))
    
    # Get balance
    de_balance = None
    if de_account:
        amount, available, limit = digital_euro_service.get_balance(de_account.holding_id)
        de_balance = {
            'amount': amount,
            'available': available,
            'limit': limit,
            'euban': de_account.euban
        }
    
    # Get trading accounts
    trading_accounts = []
    selected_account_id = request.args.get('account')
    if de_account:
        trading_accounts = marketplace_wallet_service.get_organization_trading_accounts(organization)
        if not selected_account_id and trading_accounts:
            selected_account_id = trading_accounts[0]['tid']
    
    # Get selected account details
    selected_account = None
    fame_balance = None
    fame_transactions = []
    
    if selected_account_id:
        selected_account = next((acc for acc in trading_accounts if acc['tid'] == selected_account_id), None)
        if selected_account:
            balance_result = marketplace_wallet_service.get_balance(selected_account_id)
            
            if balance_result.get('status') == 'success':
                fame_balance = {
                    'balance': balance_result.get('balance', 0.0), 
                    'status': balance_result.get('account_status', 'unknown')
                }
            
            tx_result = marketplace_wallet_service.get_transactions(selected_account_id, limit=20)
            if tx_result.get('status') == 'success':
                fame_transactions = tx_result.get('transactions', [])
    
    # Local transactions
    local_transactions = WalletTransaction.query.filter_by(
        organization_id=organization.id
    ).order_by(WalletTransaction.created_at.desc()).limit(50).all()
    
    # Render dashboard completo
    return render_template(
        'marketplace_wallet.html',
        organization=organization,
        user=user,
        de_account=de_account,
        de_balance=de_balance,
        trading_accounts=trading_accounts,
        selected_account=selected_account,
        fame_balance=fame_balance,
        fame_transactions=fame_transactions,
        local_transactions=local_transactions
    )

@bp.route('/link-digital-euro', methods=['POST'])
@login_required
def link_digital_euro():
    """Create Digital Euro account for organization"""
    logger.info("=== Starting Digital Euro account creation ===")
    
    user = get_current_user()
    organization = get_current_organization()
    
    logger.info(f"User: {user.email if user else 'None'}")
    logger.info(f"Organization: {organization.legal_name if organization else 'None'}")
    
    if not organization:
        flash('Select an organization first', 'warning')
        return redirect(url_for('auth.control_panel'))
    
    # Check if account already exists
    existing = DigitalEuroAccount.query.filter_by(organization_id=organization.id).first()
    
    if existing:
        logger.warning(f"Organization {organization.legal_name} already has DE account")
        flash('Organization already has Digital Euro account', 'warning')
        return redirect(url_for('wallet.dashboard'))
    
    try:
        # Create holding via mock service
        logger.info("Creating Digital Euro holding...")
        holding_result = digital_euro_service.create_holding(holding_type='merchant')  # lowercase!
        logger.info(f"Holding result: {holding_result}")
        
        if 'error' in holding_result:
            logger.error(f"Holding creation failed: {holding_result['error']}")
            flash(f'Failed to create Digital Euro account: {holding_result["error"]}', 'error')
            return redirect(url_for('wallet.index'))
        
        holding_id = holding_result.get('entry')
        
        if not holding_id:
            logger.error("No holding_id in response")
            flash('Invalid response from Digital Euro service', 'error')
            return redirect(url_for('wallet.index'))
        
        # Generate EUBAN
        euban = digital_euro_service.generate_euban()
        logger.info(f"Generated EUBAN: {euban}")
        
        # Save to database
        de_account = DigitalEuroAccount(
            organization_id=organization.id,
            holding_id=holding_id,
            euban=euban,
            holding_type='MERCHANT', # uppercase in DB
            created_by_user_id=user.id,
            is_active=True
        )
        
        db.session.add(de_account)
        db.session.commit()
        
        logger.info(f"✅ Created DE account {euban} for {organization.legal_name}")
        flash('✅ Digital Euro account created successfully!', 'success')
        
        # Redirect al dashboard
        return redirect(url_for('wallet.dashboard'))
        
    except Exception as e:
        logger.error(f"❌ Error creating DE account: {str(e)}")
        logger.exception(e)
        db.session.rollback()
        flash(f'An error occurred: {str(e)}', 'error')
        return redirect(url_for('wallet.index'))



@bp.route('/purchase-tokens', methods=['POST'])
@login_required
def purchase_tokens():
    """Purchase FAME tokens"""
    user = get_current_user()
    organization = get_current_organization()
    
    if not organization:
        return jsonify({'error': 'No organization selected', 'status': 'error'}), 400
    
    try:
        data = request.get_json()
        amount = data.get('amount', 0)
        trading_account_id = data.get('trading_account_id')
        
        # VALIDATION: Only whole numbers
        try:
            amount = int(amount)
        except (ValueError, TypeError):
            return jsonify({'error': 'Amount must be a whole number', 'status': 'error'}), 400
        
        if amount <= 0:
            return jsonify({'error': 'Amount must be greater than zero', 'status': 'error'}), 400
        
        if not trading_account_id:
            return jsonify({'error': 'Trading account not specified', 'status': 'error'}), 400
        
        trading_account_owner = None
        trading_accounts = marketplace_wallet_service.get_organization_trading_accounts(organization)
        
        for account in trading_accounts:
            if account.get('tid') == trading_account_id:
                trading_account_owner = account.get('owningUser')
                break
        
        result = marketplace_wallet_service.purchase_tokens(
            user=user,
            organization=organization,
            trading_account_id=trading_account_id,
            amount=float(amount),
            trading_account_owner=trading_account_owner
        )
        
        if result.get('status') == 'success':
            return jsonify(result), 200
        else:
            return jsonify(result), 400
            
    except Exception as e:
        logger.error(f"Error: {str(e)}")
        return jsonify({'error': 'Internal error', 'status': 'error'}), 500


@bp.route('/redeem-tokens', methods=['POST'])
@login_required
def redeem_tokens():
    """Redeem FAME tokens"""
    user = get_current_user()
    organization = get_current_organization()
    
    if not organization:
        return jsonify({'error': 'No organization selected', 'status': 'error'}), 400
    
    try:
        data = request.get_json()
        amount = data.get('amount', 0)
        trading_account_id = data.get('trading_account_id')
        
        # VALIDATION: Only whole numbers
        try:
            amount = int(amount)
        except (ValueError, TypeError):
            return jsonify({'error': 'Amount must be a whole number', 'status': 'error'}), 400
        
        if amount <= 0:
            return jsonify({'error': 'Amount must be greater than zero', 'status': 'error'}), 400
        
        if not trading_account_id:
            return jsonify({'error': 'Trading account not specified', 'status': 'error'}), 400
        
        trading_account_owner = None
        trading_accounts = marketplace_wallet_service.get_organization_trading_accounts(organization)
        
        for account in trading_accounts:
            if account.get('tid') == trading_account_id:
                trading_account_owner = account.get('owningUser')  # ← UID
                break
        
        result = marketplace_wallet_service.redeem_tokens(
            user=user,
            organization=organization,
            trading_account_id=trading_account_id,
            amount=float(amount),
            trading_account_owner=trading_account_owner
        )
        
        if result.get('status') == 'success':
            return jsonify(result), 200
        else:
            return jsonify(result), 400
            
    except Exception as e:
        logger.error(f"Error: {str(e)}")
        return jsonify({'error': 'Internal error', 'status': 'error'}), 500


@bp.route('/api/balances')
@login_required
def api_balances():
    """Get current balances"""
    organization = get_current_organization()
    
    if not organization:
        return jsonify({'error': 'No organization selected'}), 400
    
    de_account = DigitalEuroAccount.query.filter_by(
        organization_id=organization.id,
        is_active=True
    ).first()
    
    if not de_account:
        return jsonify({'error': 'No Digital Euro account'}), 404
    
    de_amount, de_available, de_limit = digital_euro_service.get_balance(de_account.holding_id)
    
    trading_accounts = marketplace_wallet_service.get_organization_trading_accounts(organization)
    accounts_with_balance = []
    
    for account in trading_accounts:
        balance_result = marketplace_wallet_service.get_balance(account['tid'])
        if balance_result.get('status') == 'success':
            accounts_with_balance.append({
                'tid': account['tid'],
                'owner': account['user_email'],
                'balance': balance_result.get('balance', 0.0)
            })
    
    return jsonify({
        'digital_euro': {
            'amount': de_amount,
            'available': de_available,
            'limit': de_limit
        },
        'trading_accounts': accounts_with_balance
    })
